<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Carbon\Carbon;
use Google_Client;
use Google_Service_Drive;
use Google_Service_Drive_DriveFile;
use Storage;

class DatabaseBackupToDrive extends Command
{

    protected $signature = 'backup:db-drive';
    protected $description = 'Backup database and upload to Google Drive';

    public function handle()
    {
        // === CONFIGURATION ===
        $dbUser = env('DB_USERNAME');
        $dbPass = env('DB_PASSWORD');
        $dbName = env('DB_DATABASE');
        $backupPath = storage_path('app/backups');
        $date = date('Y-m-d_H-i-s');
        $filename = "{$dbName}_{$date}.sql.gz";
        $filePath = "$backupPath/$filename";

        if (!is_dir($backupPath)) {
            mkdir($backupPath, 0755, true);
        }

        // === Dump database ===
        $command = "mysqldump -u$dbUser -p$dbPass $dbName | gzip > $filePath";
        exec($command, $output, $return);
        if ($return !== 0) {
            $this->error('Database backup failed.');
            return 1;
        }

        // === Google Client setup ===
        $client = new Google_Client();
        $client->setApplicationName('Laravel DB Backup');
        $client->setScopes([Google_Service_Drive::DRIVE_FILE]);
        $client->setAuthConfig(storage_path('app/credentials.json'));
        $client->setAccessType('offline');

        $tokenPath = storage_path('app/token.json');
        if (file_exists($tokenPath)) {
            $accessToken = json_decode(file_get_contents($tokenPath), true);
            $client->setAccessToken($accessToken);
        }

        if ($client->isAccessTokenExpired()) {
            if ($client->getRefreshToken()) {
                $client->fetchAccessTokenWithRefreshToken($client->getRefreshToken());
            } else {
                $authUrl = $client->createAuthUrl();
                $this->info("1st Time Auth Required: Open this URL and paste the code here:");
                $this->info($authUrl);
                $code = trim(fgets(STDIN));
                $accessToken = $client->fetchAccessTokenWithAuthCode($code);
                file_put_contents($tokenPath, json_encode($accessToken));
                $client->setAccessToken($accessToken);
            }
        }

        $service = new Google_Service_Drive($client);

        // === Upload to Google Drive ===
        $fileMetadata = new Google_Service_Drive_DriveFile([
            'name' => $filename,
            // 'parents' => ['your_drive_folder_id'] // optional
        ]);

        $content = file_get_contents($filePath);

        $file = $service->files->create($fileMetadata, [
            'data' => $content,
            'mimeType' => 'application/gzip',
            'uploadType' => 'multipart',
            'fields' => 'id',
        ]);

        $this->info("✅ Backup uploaded to Google Drive. File ID: " . $file->id);

        // Optional: Delete local backup after upload
        unlink($filePath);

        return 0;
    }
   

}
