<?php

namespace App\Console\Commands;

use App\Models\Edu\Device\DeviceLog;
use App\Models\Edu\Issue\IssueDetail;
use App\Traits\RequestHandlingWithMachine;
use App\Traits\TimeRelatedFunctionTrait;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;

class FetchDeviceLogByAccessIdCommand extends Command
{
    use RequestHandlingWithMachine,TimeRelatedFunctionTrait;

    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'fetch:log';


    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Fetched Device Log';


    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        set_time_limit(500);
        date_default_timezone_set('Asia/Dhaka');

        $issue = new IssueDetail();
        $issue->issue_type = 'fetch:log';
        $issue->issuer_type = 'edu';
        $issue->issue_details = "Started fetch:log";
        $issue->save();

       $now = Carbon::now();

        if ($now->format('H:i') === '23:50') {
            $start = Carbon::today()->setTime(1, 15, 0);   // 01:15:00
            $end = Carbon::today()->setTime(23, 50, 0);    // 23:50:00

        } else {
            // Run last 2-hour range
            $start = $now->copy()->subMinutes(120);
            $end = $now;
        }

        $start_date = $start->toDateString();
        $end_date = $end->toDateString();
        $s_time = $start->format('H:i:s');
        $e_time = $end->format('H:i:s');

        // Log info
        Log::info('Schedule Started', [
            'from' => $start->format('Y-m-d H:i:s'),
            'to'   => $end->format('Y-m-d H:i:s'),
        ]);

        // Stellar
        $this->stellarDeviceLogs($start_date, $s_time, $end_date, $e_time);

        // Inovace
        $this->inovaceDeviceLogs($start_date, $s_time, $end_date, $e_time);

    }
    

    private function stellarDeviceLogs($start_date, $s_time, $end_date, $e_time)
        {
            $deviceLog = new DeviceLog();
            $logs = $this->fetchLogByDevice($start_date, $s_time, $end_date, $e_time);

            if (!is_array($logs)) {
                \Log::warning('fetchLogByDevice returned non-array', ['logs' => $logs]);
                return;
            }
            

            if (count($logs) > 0) {
                $accessIds = collect($logs)->pluck('access_id')->toArray();

                $existing = [];
                foreach (array_chunk($accessIds, 500) as $chunk) {
                    $existingChunk = $deviceLog->whereIn('log_id', $chunk)->pluck('log_id')->toArray();
                    $existing = array_merge($existing, $existingChunk);
                }
                $existing = array_unique($existing);

                $newLogs = [];
                foreach ($logs as $v) {
                    if (in_array($v->access_id, $existing)) continue;

                    $newLogs[] = [
                        'log_id'             => $v->access_id,
                        'company'            => 'stellar',
                        'device_id'          => $v->unit_id,
                        'user_biometric_id'  => $v->registration_id,
                        'access_time'        => $v->access_time,
                        'access_date'        => $v->access_date,
                        'access_date_time'   => $v->access_date . ' ' . $v->access_time,
                        'created_at'         => now(),
                        'updated_at'         => now(),
                    ];
                }

                foreach (array_chunk($newLogs, 500) as $chunk) {
                    DeviceLog::insert($chunk);
                }
            }
        }

 
    private function inovaceDeviceLogs($start_date, $s_time, $end_date, $e_time)
        {
            $deviceLog = new DeviceLog();
            $logs = $this->fetchInovaceLog($start_date, $s_time, $end_date, $e_time);

             if (!is_array($logs)) {
                \Log::warning('fetchInovaceLog returned non-array', ['logs' => $logs]);
                return;
            }

            if (count($logs) > 0) {
                $accessIds = collect($logs)->pluck('uid')->toArray();

                $existing = [];
                foreach (array_chunk($accessIds, 500) as $chunk) {
                    $existingChunk = $deviceLog->whereIn('log_id', $chunk)->pluck('log_id')->toArray();
                    $existing = array_merge($existing, $existingChunk);
                }
                $existing = array_unique($existing);

                $newLogs = [];
                foreach ($logs as $v) {
                    if (in_array($v->uid, $existing)) continue;

                    $newLogs[] = [
                        'log_id'             => $v->uid,
                        'company'            => 'inovace',
                        'device_id'          => $v->device_identifier,
                        'user_biometric_id'  => $v->person_identifier,
                        'access_time'        => date('H:i:s', strtotime($v->logged_time)),
                        'access_date'        => date('Y-m-d', strtotime($v->logged_time)),
                        'access_date_time'   => $v->logged_time,
                        'created_at'         => now(),
                        'updated_at'         => now(),
                    ];
                }

                foreach (array_chunk($newLogs, 500) as $chunk) {
                    DeviceLog::insert($chunk);
                }
            }
        }


}