<?php

namespace App\Http\Controllers\Edu\SiteGroup;

use App\Http\Controllers\ProjectController;
use App\Mail\SendMailToUserWithQueue;
use App\Models\Academic\AcademicClass;
use App\Models\Academic\AcademicDepartment;
use App\Models\Academic\AcademicGroup;
use App\Models\Edu\Issue\IssueDetail;
use App\Models\Modules\SubModule;
use App\Models\Package\EduPackage;
use App\Models\Site\SiteGroupSetting;
use App\Models\Site\SiteInfo;
use App\Models\SiteGroup\GlobalSiteGroup;
use App\Models\SiteGroup\Role\GlobalSiteGroupRole;
use App\Models\User\GlobalUser;
use App\Traits\AliasUniqueCheck;
use Auth;
use Error;
use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\ValidationException;
use Image;
use Log;
use Mail;
use Response;
use DB;

class SiteGroupController extends ProjectController
{
    use AliasUniqueCheck;
    private $siteGroup;
    private $siteInfo;
    private $eduPackage;
    private $siteGroupSetting;

    public function __construct(GlobalSiteGroup $siteGroup, SiteInfo $siteInfo, EduPackage $eduPackage, SiteGroupSetting $siteGroupSetting)
    {
        $this->middleware('auth');
        $this->middleware('eduworlduserchecker');
        $this->siteGroup = $siteGroup;
        $this->siteInfo = $siteInfo;
        $this->eduPackage = $eduPackage;
        $this->siteGroupSetting = $siteGroupSetting;

    }

    public function index()
    {
        $viewType = 'Institution Group List';
        return view('default.admin.layouts.master', compact('viewType'));
        return view('default.admin.site_group.index', compact('viewType'));

    }



    public function assignModule($id = null)
    {
        $viewType = 'Assign Module To Group';
        return view('default.admin.site_group.assign-module', compact('viewType'));
    }

    public function createSiteGroup(Request $validatedRequest)
    {
        try {
            $this->validate($validatedRequest, [
                'name' => 'required|unique:site_groups',  //unique:table name
                'group_alias' => 'required|unique:site_groups,group_alias|unique:site_infos,site_alias',
                'group_email' => 'required|email|unique:site_groups',
                'group_address' => 'required',
                'group_phone' => 'required|numeric|unique:site_groups',
                'status' => 'required',
                'academic_group_ids' => 'required',

                //'group_logo'=>'required|regex:/^data:image/'
                // 'required|image|mimes:jpeg,png,jpg,gif,svg|max:1024'
                //'group_logo'=>'required|regex:/^data:image/'
                // 'required|image|mimes:jpeg,png,jpg,gif,svg|max:1024'
            ]);


            if ($this->aliasAlreadyExisted($validatedRequest->group_alias)) {
                throw new Error('The Alias Already Existed');
            }


            if ($validatedRequest->group_logo == "demo_logo.jpg") {
                $this->siteGroup->group_logo = $validatedRequest->group_logo;

            } else {

                $name = time() . '.' . explode('/', explode(':', substr($validatedRequest->group_logo, 0, strpos($validatedRequest->group_logo, ';')))[1])[1];
                Image::make($validatedRequest->group_logo)->save(public_path('img/site_group/logo/') . $name);
                $validatedRequest->merge(['group_logo' => $name]);
                $userPhoto = public_path('img/site_group/logo/');
                if (file_exists($userPhoto)) {
                    @unlink($userPhoto);
                }
                $this->siteGroup->group_logo = $name;
            }

            if (count($this->siteGroup->ownFields) > 0) {
                foreach ($this->siteGroup->ownFields as $ownField) {
                    if ($validatedRequest->{$ownField}) {
                        $this->siteGroup->{$ownField} = $validatedRequest->{$ownField};
                    }
                }
                $this->siteGroup->status = $validatedRequest->status;
            }

            if ($this->siteGroup->save()) {
                if ($this->createRole($this->siteGroup)) {

                    if ($this->createAdminUser($this->siteGroup)) {
                        return Response::json(['message' => 'Site Group Save Successfully with all!', 'mode' => 'Saved!'], 200);
                    } else {
                        return Response::json(['message' => 'Site Group Save Successfully without creating admin user!', 'mode' => 'Saved!'], 200);
                    }
                } else {
                    return Response::json(['message' => 'Site Group Save Successfully Without creating role!', 'mode' => 'Saved!'], 200);

                }


            }
        } catch (ValidationException $exception) {
            return JsonResponse::create(['message' => $exception->getMessage(), 'errors' => $exception->validator->getMessageBag()->toArray()], 422);
        } catch (Exception $exception) {
            return JsonResponse::create(['message' => $exception->getMessage()], 500);

        }
    }

    public function editSiteGroupForm($id)
    {
        $viewType = 'Edit Institution Group';
        $siteGroupToEdit = $this->editFormModel($this->siteGroup->findOrFail($id));
        return view('default.admin.site_group.edit', compact('siteGroupToEdit', 'viewType'));
    }

    public function editSiteGroup(Request $validatedRequest, $id)
    {

        $this->validate($validatedRequest, [
            'academic_group_ids' => 'required',
            'name' => 'required|unique:site_groups,id,' . $id,
            'group_alias' => 'required|unique:site_groups,group_alias,' . $id . ',id|unique:site_infos,site_alias',
            'group_email' => 'required|email|unique:site_groups,id,' . $id,
            'group_address' => 'required',
            'group_phone' => 'required|numeric|unique:site_groups,group_phone,' . $id,
            'status' => 'required',
        ]);

        if ($this->aliasAlreadyExistedUpdate($validatedRequest->group_alias, $id)) {
            throw new Error('The Alias Already Existed');
        }

        $unique_academic_group_ids = null;
        if ($validatedRequest->academic_group_ids) {
            $unique_academic_group_ids = implode(',', array_unique(explode(',', $validatedRequest->academic_group_ids)));

        }

        $sitGroupToEdit = $this->siteGroup->findOrFail($id);

        if (count($sitGroupToEdit->ownFields) > 0) {
            foreach ($sitGroupToEdit->ownFields as $ownField) {
                if ($validatedRequest->{$ownField}) {
                    $sitGroupToEdit->{$ownField} = $validatedRequest->{$ownField};
                }
            }
            $sitGroupToEdit->status = $validatedRequest->status;
            $sitGroupToEdit->academic_group_ids = $unique_academic_group_ids;
        }

        $currentPhoto = $sitGroupToEdit->group_logo;

        if ($validatedRequest->group_logo == "demo_logo.jpg") {
            $sitGroupToEdit->group_logo = $validatedRequest->group_logo;
        } elseif ($validatedRequest->group_logo != $currentPhoto) {

            $name = time() . '.' . explode('/', explode(':', substr($validatedRequest->group_logo, 0, strpos($validatedRequest->group_logo, ';')))[1])[1];
            Image::make($validatedRequest->group_logo)->save(public_path('img/site_group/logo/') . $name);

            $validatedRequest->merge(['group_logo' => $name]);
            $userPhoto = public_path('img/site_group/logo/' . $currentPhoto);
            if (file_exists($userPhoto) && $currentPhoto != "demo_logo.jpg") {
                @unlink($userPhoto);
            }
            $sitGroupToEdit->group_logo = $name;
        }

        if ($sitGroupToEdit->save()) {
            if ($this->assignPackagesToSiteGroup($sitGroupToEdit, $validatedRequest->edu_package_id)) {
                return Response::json(['message' => 'Site Group Updated Successfully With assigning Package to Super Admin', 'mode' => 'Saved!'], 200);
            } else {
                return Response::json(['message' => 'Site Group Updated Successfully Without assigning Package to Admin', 'mode' => 'Saved!'], 200);
            }
        }
        return Response::json(['message' => 'Site Group Updated Successfully With only Update', 'mode' => 'Saved!'], 200);
    }


    public function createRole($newlyCreatedSiteGroup)
    {
        try {

            GlobalSiteGroupRole::firstOrCreate(
                [
                    'name' => 'Super Admin',
                    'key' => 'superadmin',
                    'status' => 1,
                    'site_group_id' => $newlyCreatedSiteGroup->id
                ]
            );
            GlobalSiteGroupRole::firstOrCreate(
                [
                    'name' => 'Employee',
                    'key' => 'employee',
                    'status' => 1,
                    'site_group_id' => $newlyCreatedSiteGroup->id
                ]
            );

            return true;
        } catch (Exception $exception) {
            $issue = new    IssueDetail();
            $issue->username = Auth::user()->username;
            $issue->issuer_type = 'edu';
            $issue->issue_type = 'sg_creation';
            $issue->issue_details = json_encode($exception->getMessage());
            $issue->save();
        }
    }


    public function createAdminUser(GlobalSiteGroup $newSiteGroup)
    {
        try {


            $global_user = new GlobalUser();
            $user_name = $newSiteGroup->id . $newSiteGroup->group_alias;
            $adminPassword = rand(100000, 200000);
            $name = $newSiteGroup->name;
            $group_email = $newSiteGroup->group_email;
            $group_alias = $newSiteGroup->group_alias;

            $global_user->username = $user_name;
            $global_user->email = $group_email;
            $global_user->password = bcrypt($adminPassword);
            $global_user->alias = $group_alias;
            $global_user->site_id = $user_name;
            $global_user->user_type = 'employee';
            $global_user->status = $newSiteGroup->status;
            $global_user->save();


            DB::table('user_translations')->insert(
                [
                    'user_id' => $global_user->id,
                    'first_name' => $name,
                    'locale' => 'en'
                ]
            );

            DB::table('photos')->insert(
                [
                    'name' => 'image-not.png',
                    'user_id' => $global_user->id,
                    'imageable_id' => $global_user->id,
                    'imageable_type' => 'App\User'
                ]
            );

            $this->setAdminRole($global_user->id, $newSiteGroup);

            $sub_domain = $this->getDomain($newSiteGroup->group_alias);

            $data = [
                'username' => $user_name,
                'name' => $name,
                'sub_domain' => $sub_domain,
                'password' => $adminPassword,
                'message' => 'Site Group Successfully Created!',
                'user_id' => Auth::user()->username
            ];
            $subject = "New Site Group Create";
            $mail = $group_email ?? 'edudeskbd@gmail.com';

            Mail::to(['eduworlderp@gmail.com', $mail])->send(new SendMailToUserWithQueue($subject, $data));

            return true;
        } catch (Exception $exception) {

            $issue = new    IssueDetail();
            $issue->username = Auth::user()->username;
            $issue->issuer_type = 'edu';
            $issue->issue_type = 'sg_creation';
            $issue->issue_details = json_encode($exception->getMessage());
            $issue->save();

            return false;
        }

    }

    private function setAdminRole($userId, GlobalSiteGroup $newlyCreatedSiteGroup)
    {
        try {
            $role = GlobalSiteGroupRole::firstOrCreate(
                [
                    'name' => 'Super Admin',
                    'key' => 'superadmin',
                    'status' => 1,
                    'site_group_id' => $newlyCreatedSiteGroup->id
                ]
            );
            if ($role) {
                $roleId = $role->id;
                DB::table('site_group_role_user')->insert(
                    [
                        'user_id' => $userId,
                        'site_group_role_id' => $roleId
                    ]
                );
            }
            $this->assignPackagesToSiteGroup($newlyCreatedSiteGroup, $newlyCreatedSiteGroup->edu_package_id);
            return true;
        } catch (Exception $exception) {

            $issue = new    IssueDetail();
            $issue->username = Auth::user()->username;
            $issue->issuer_type = 'edu';
            $issue->issue_type = 'sg_creation';
            $issue->issue_details = json_encode($exception->getMessage());
            $issue->save();
            return false;

        }
    }


    public function assignPackagesToSiteGroup(GlobalSiteGroup $site_group, $package_id)
    {
        try {
            $sub_module_ids = [];
            $package = $this->eduPackage->findOrFail($package_id);
            if ($package) {

                $module_ids = $package->modules->pluck('id')->toArray();
                $unique_module_ids = array_unique($module_ids);
                foreach ($unique_module_ids as $k => $v) {
                    $sub_module_ids = array_merge($sub_module_ids, SubModule::whereModuleId($v)->pluck('id')->toArray());
                }
                $unique_sub_module_ids = array_unique($sub_module_ids);


                $site_group->subModules()->sync($unique_sub_module_ids);
                $site_group->save();

                //assigned newly subModule to superAdmin too

                if (file_exists(base_path("resources/views/default/sidebars/" . $site_group->group_alias))) {
                    deleteDir((base_path("resources/views/default/sidebars/" . $site_group->group_alias)));
                }

                $site_group_role = GlobalSiteGroupRole::firstOrCreate(
                    [
                        'name' => 'Super Admin',
                        'key' => 'superadmin',
                        'status' => 1,
                    ]);


                $imploded_sub_module_ids = implode(',', $sub_module_ids) ?? null;

                // throw new Error( $site_group_id_obj->id);

                if ($site_group_role->roleAccess == null) {
                    $site_group_role->roleAccess()->insert([
                        'site_group_role_id' => $site_group_role->id,
                        'site_group_id' => $site_group->id,
                        'sub_module_ids' => $imploded_sub_module_ids,
                    ]);
                } else {
                    $site_group_role->roleAccess->update([
                        'site_group_id' => $site_group->id,
                        'sub_module_ids' => $imploded_sub_module_ids
                    ]);
                }
            }
            return true;

        } catch (Exception $exception) {
            Log::info($exception->getMessage());

            $issue = new    IssueDetail();
            $issue->username = Auth::user()->username;
            $issue->issuer_type = 'edu';
            $issue->issue_type = 'sg_creation';
            $issue->issue_details = json_encode($exception->getMessage());
            $issue->save();

            return false;
        }
    }


    public function getSiteGroupList()
    {
        $siteGroups = $this->siteGroup->paginate(50);
        $academicGroups = AcademicGroup::all();//  DB::select('SELECT id, academic_group_name FROM academic_groups');
        return [$siteGroups, $academicGroups];

    }


    public function getSiteGroupFullList()
    {
   //     dd('fsfsf');
        $data = $this->siteGroup->all();

        return response()->json($data, 200);


    }

    public function getSiteGroupListWithSites()
    {
        $data = $this->siteGroup->with('sites')->get();
        return response()->json($data, 200, [], JSON_NUMERIC_CHECK);
    }

}
