<?php

namespace App\Http\Controllers\Site\Employee;

use App\Forms\DataHelper;
use App\Forms\FormControll;
use App\Models\Academic\AcademicClass;
use App\Models\Academic\AcademicDepartment;
use App\Models\Academic\AcademicShift;
use App\Models\Academic\AcademicVersion;
use App\Models\AcademicYear\AcademicYear;
use App\Models\Employee\SiteEmployeeDepartmentClassSubject;
use App\Models\Site\Employee\Employee;
use App\Models\Site\SiteInfoSetting\SiteInfoSetting;
use App\Models\Site\SiteShiftClassGroupSectionSetting;
use App\Models\User\GlobalUser;
use App\Traits\Site\Routine\RoutineFunction;
use App\User;
use App\Models\Event\Event;
use Error;
use Exception;
use App\Http\Controllers\Controller;
use App\Models\Academic\AcademicLeaveType;
use App\Models\Academic\AcademicSubject;
use App\Models\Employee\EmployeeEducationBackground;
use App\Models\EmploymentHistory\EmploymentHistory;
use App\Models\Site\Employee\EmployeeAttendance;
use App\Models\Site\EmployeeLeave\SiteEmployeeLeaveGenerate;
use App\Models\Site\EmployeeLeave\SiteEmployeeLeaveHistory;
use App\Models\Site\EmployeeLeave\SiteLeavePolicy;
use App\Models\Site\EmployeOnDutyInHolidaySetting\EmployeOnDutyInHolidaySetting;
use App\Models\Site\SiteInfo;
use App\Models\SiteGroup\SiteGroupModel;
use App\Models\User\EmployeeHistory;
use Carbon\Carbon;
use Carbon\CarbonPeriod;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Response;
use Session;
use PDF;
use Illuminate\Support\Str;
use Mpdf\Tag\Dd;
use View;

class EmployeeController extends Controller
{

    use  FormControll, DataHelper, RoutineFunction;


    private $SiteShiftClassGroupSectionSetting;
    private $employee;

    public function __construct(SiteShiftClassGroupSectionSetting $SiteShiftClassGroupSectionSetting, Employee $employee)
    {
        $this->middleware('auth');
        $this->middleware('sitepagechecker');

        $this->SiteShiftClassGroupSectionSetting = $SiteShiftClassGroupSectionSetting;
        $this->employee = $employee;
    }


    public function updateFieldData(Request $request)
    {
        try {

            $the_emp = $this->employee->findOrFail($request->id);

            if (count($the_emp->saveFields) > 0) {
                foreach ($the_emp->saveFields as $field) {
                    if ($request->{$field}) {
                        $the_emp->{$field} = $request->{$field};
                    }
                }
            }


            if ($the_emp->save()) {
                return Response::json(['message' => 'Successfully Updated The Employee!', 'mode' => 'Update'], 200);
            }
        } catch (Exception $exception) {
            return JsonResponse::create(['message' => $exception->getMessage()], 410);
        }
    }


    // payroll
    private function calculateDeductions($payroll)
    {
        $SalaryofOneDay = $payroll->base_salary / 30;


        // Calculate late deductions
        if ($payroll->late_status_count) {
            $total_late = $payroll->total_late;
            if ($total_late >= $payroll->countable_late && $payroll->countable_late != 0) {
                $countable_late_result = floor($total_late / $payroll->countable_late);
                $total_late_deduction = $SalaryofOneDay * $countable_late_result;
            } else {
                $total_late_deduction = 0;
            }
        } else {
            $total_late_deduction = 0;
        }

        // Calculate absent deductions
        if ($payroll->absent_count) {
            $total_absent = $payroll->total_absent;
            if ($total_absent >= $payroll->countable_absent && $payroll->countable_absent != 0) {
                $countable_absent_result = floor($total_absent / $payroll->countable_absent);
                $total_absent_deduction = $SalaryofOneDay * $countable_absent_result;
            } else {
                $total_absent_deduction = 0;
            }
        } else {
            $total_absent_deduction = 0;
        }

        // Calculate unpaid leave deductions
        if ($payroll->unpaid_leave_count) {
            $total_unpaid_leave = $payroll->total_unpaid_leave;
            if ($total_unpaid_leave >= $payroll->countable_unpaid_leave && $payroll->countable_unpaid_leave != 0) {
                $countable_unpaid_result = floor($total_unpaid_leave / $payroll->countable_unpaid_leave);
                $total_unpaid_leave_deduction = $SalaryofOneDay * $countable_unpaid_result;
            } else {
                $total_unpaid_leave_deduction = 0;
            }
        } else {
            $total_unpaid_leave_deduction = 0;
        }

        // Calculate Half Day leave deductions
        if ($payroll->half_day_leave_count) {
            $total_half_day_leave = $payroll->total_half_day_leave;
            if ($total_half_day_leave >= $payroll->countable_half_day_leave && $payroll->countable_half_day_leave != 0) {

                $countable_half_result = floor($total_half_day_leave / $payroll->countable_half_day_leave);
                $total_half_day_leave_deduction = $SalaryofOneDay * $countable_half_result;
            } else {
                $total_half_day_leave_deduction = 0;
            }
        } else {
            $total_half_day_leave_deduction = 0;
        }


        return [
            'SalaryofOneDay'            => $SalaryofOneDay,
            'total_late_deduction'      => $total_late_deduction,
            'total_absent_deduction'    => $total_absent_deduction,
            'total_unpaid_leave_deduction' => $total_unpaid_leave_deduction,
            'total_half_day_leave_deduction' => $total_half_day_leave_deduction
        ];
    }
    public function getAllEmployeeAttendance(Request $request)
    {

        try {


            $year = $request->selectedYear;
            $month = $request->selectedMonth;

            $user_type = $request->types ?? 'employee';


            $search_txt     = $request->search_txt;
            $paginate       = (int)$request->paginate;
            $status         = $request->status;
            $finger_taken   = $request->finger_taken;
            $attendance_sms = $request->attendance_sms;
            $is_teacher     = $request->is_teacher ?? [0, 1]; //



            if (!$user_type) {
                throw new Exception('Check At Least One Type!' . $user_type);
            }

            $txt = '%' . $search_txt . '%';

            $employees = $this->employee
                ->with('siteRoles')
                ->with(['latestPhoto'])
                ->with(['designation'])
                ->select('id', 'username', 'user_type', 'email', 'contact_number', 'status',  'designation_id')
                ->where('user_type', $user_type)
                ->when($status == 1, function ($q) use ($status) {
                    return $q->whereHas('activeHistory');
                })
                ->with([
                    'attendances' => function ($query) use ($year, $month) {
                        $query->select(
                            'user_id',
                            DB::raw("count(*) as total_attendence"),
                            DB::raw("SUM(late_status) as total_late_status"),
                        )
                            ->where('present_year', $year)
                            ->where('present_month', $month)
                            ->groupBy('user_id');
                    }
                ])
                ->with([
                    'EmployeeAttendance' => function ($query) use ($year, $month) {
                        $query->select(
                            'user_id',
                            'manually_attendance',
                            'countable_absent',
                            'countable_unpaid_leave',
                            'countable_half_day_leave',
                            'countable_late',
                        )
                            ->where('year', $year)
                            ->where('month', $month);
                    }
                ])
                ->whereHas('empHistories', function ($q1) use ($is_teacher) {
                    return $q1->whereIn('is_teacher', $is_teacher);
                })
                ->when($status < 2, function ($q) use ($status) {
                    return $q->where('status', $status);
                })
                ->when($status == 2, function ($q) {
                    return $q->whereNotNull('deleted_at');
                })

                ->when($search_txt, function ($q) use ($txt) {
                    return $q->where(function ($query) use ($txt) {
                        $query->where('username', 'LIKE', $txt)
                            ->orWhere('email', 'LIKE', $txt)
                            ->orWhere('contact_number', 'LIKE', $txt)
                            ->orWhereTranslationLike('first_name', $txt)
                            ->orWhereTranslationLike('last_name', $txt);
                    });
                })
                // ->where('id', 53345)
                ->orderBy('employee_serial', 'ASC')
                ->get();

            $firstDayOfMonth = mktime(0, 0, 0, $month, 1, $year);

            $TotlaDayOfMonth = date('t', $firstDayOfMonth);

            $offDayOfMonth = Event::whereYear('activate_date', '=', $year)
                ->whereMonth('activate_date', '=', $month)
                ->where('status', 1)
                ->get();

            // Start date is the first day of the month
            $search_start_date = Carbon::createFromDate($year, $month, 1)->startOfMonth()->format('Y-m-d');

            // End date is the last day of the month
            $search_end_date = Carbon::createFromDate($year, $month, 1)->endOfMonth()->format('Y-m-d');


            $employees->transform(function ($value) use ($offDayOfMonth, $TotlaDayOfMonth, $search_start_date, $search_end_date, $year, $month) {

                $total_without_unpaid_leave = 0;
                $total_unpaid_leave         = 0;
                $total_half_day_leave         = 0;
                $total_leave = 0;

                $MonthYearWiseEmployeeAttendance = EmployeeAttendance::select('id', 'user_id', 'present_date', 'present_month', 'present_year')->where('user_id', $value->id)
                    ->where('present_month', $month)
                    ->where('present_year', $year)
                    ->get();

                // Get employee history
                $employee_history = EmployeeHistory::where('user_id', $value->id)
                    ->where('status', 1)
                    ->orderBy('id', 'DESC')
                    ->first();

                // Get leave history
                $leave_history = SiteEmployeeLeaveHistory::where('employee_history_id', $employee_history->id)
                    ->with([
                        'siteEmployeeLeaveGenerate',
                        'siteEmployeeLeaveGenerate.siteLeavePolicy',
                        'siteEmployeeLeaveGenerate.siteLeavePolicy.academicLeaveType',

                    ])
                    ->where('status', 1)
                    ->where('leave_status', 'approved')
                    ->get();



                if ($leave_history->count()) {
                    // Loop through the leave history
                    foreach ($leave_history as $leave) {


                        $start_date = $leave->from_date;
                        $end_date = $leave->to_date;

                        // Generate the period for the leave range
                        $leave_period = CarbonPeriod::create($start_date, $end_date);

                        // Generate the search date period (range)
                        $search_period = CarbonPeriod::create($search_start_date, $search_end_date);

                        // Loop through each date in the search range
                        foreach ($search_period as $search_date) {
                            $search_formatted_date = $search_date->format('Y-m-d');

                            // Check if the leave date exists within the search range
                            foreach ($leave_period as $leave_date) {
                                $leave_formatted_date = $leave_date->format('Y-m-d');

                                if ($leave_formatted_date == $search_formatted_date) {
                                    // Increment total leave counter
                                    $total_leave++;

                                    $site_leave_policy =  SiteEmployeeLeaveGenerate::where('id', $leave->site_employee_leave_generate_id)->first();
                                    $SiteLeavePolicy   =  SiteLeavePolicy::where('id', $site_leave_policy->site_leave_policy_id)->first();
                                    $AcademicLeaveType =  AcademicLeaveType::where('id', $SiteLeavePolicy->academic_leave_type_id)->first();
                                    //check unpaid leave
                                    if ($AcademicLeaveType->short_name ==  'UP') {
                                        $total_unpaid_leave++;
                                    } elseif ($AcademicLeaveType->short_name ==  'HDL') {
                                        $total_half_day_leave++;
                                    } elseif ($AcademicLeaveType->short_name !=  'UP') {
                                        $total_without_unpaid_leave++;
                                    }


                                    // break; // No need to continue checking once a match is found
                                }
                            }
                        }
                    }
                }

                $EmployeOnDutyInHolidaySettings = EmployeOnDutyInHolidaySetting::where('employe_designation', $value->designation->name)
                    ->where('year', $year)
                    ->where('month', $month)
                    ->get();

                // Count total days in offDayOfMonth and EmployeOnDutyInHolidaySettings
                $EmployeOnDutyInHolidaySettingCount = $EmployeOnDutyInHolidaySettings->count();
                $offDayOfMonthCount = $offDayOfMonth->count();

                // Filter the offDayOfMonth to exclude entries in EmployeOnDutyInHolidaySettings
                $filteredoffDayOfMonth = $offDayOfMonth->filter(function ($offDay) use ($EmployeOnDutyInHolidaySettings) {
                    return !$EmployeOnDutyInHolidaySettings->contains(function ($employeOnDuty) use ($offDay) {
                        return $offDay->activate_date === $employeOnDuty->on_duty;
                    });
                });

                // Filter attendance based on filtered off days
                $matchedAttendancePresentCount = $MonthYearWiseEmployeeAttendance->filter(function ($attendance) use ($filteredoffDayOfMonth) {
                    return $filteredoffDayOfMonth->contains(function ($offDay) use ($attendance) {
                        return $attendance->present_date === $offDay->activate_date;
                    });
                })->count();

                // Subtract the count of EmployeOnDutyInHolidaySetting from offDayOfMonth count
                $TotalOffDayOfMonth = $offDayOfMonthCount - ($EmployeOnDutyInHolidaySettingCount + $matchedAttendancePresentCount);


                $EmployeeAttendance        = $value->EmployeeAttendance->first();

                $EmployeeDeviceAttendance  = $value->attendances->first();


                $total_device_attendence   = $EmployeeDeviceAttendance ? $EmployeeDeviceAttendance->total_attendence : 0;
                $total_late_status         = $EmployeeDeviceAttendance ? $EmployeeDeviceAttendance->total_late_status : "0";

                $TotalAcademicDay          = $TotlaDayOfMonth - $TotalOffDayOfMonth;
                $total_present = ($total_device_attendence  + $TotalOffDayOfMonth + ($total_without_unpaid_leave ?? 0) + ($EmployeeAttendance ? $EmployeeAttendance->manually_attendance : 0) ?? null);

                if ($TotlaDayOfMonth < $total_present) {
                    $present_deduct = $total_present - $TotlaDayOfMonth;
                    $total_device_attendence = $total_device_attendence - $present_deduct;
                }


                // dd($TotlaDayOfMonth - ($total_device_attendence  + $TotalOffDayOfMonth + ($total_without_unpaid_leave ?? 0) + ($EmployeeAttendance ? $EmployeeAttendance->manually_attendance : 0) ?? null));



                return [
                    'id'                             => $value->id,
                    'full_name'                      => $value->full_name,
                    'designation'                    => $value->designation->name,
                    'total_device_attendence'        => $total_device_attendence,
                    'total_present'                  => $total_present,
                    'offDayOfMonth'                  => $offDayOfMonthCount - ($EmployeOnDutyInHolidaySettingCount + $matchedAttendancePresentCount) ?? 0,
                    'employee_manually_attendance'   => $EmployeeAttendance ? $EmployeeAttendance->manually_attendance : 0,
                    'countable_unpaid_leave'         => $EmployeeAttendance ? $EmployeeAttendance->countable_unpaid_leave : 0,
                    'countable_half_day_leave'         => $EmployeeAttendance ? $EmployeeAttendance->countable_half_day_leave : 0,
                    'countable_absent'               => $EmployeeAttendance ? $EmployeeAttendance->countable_absent : 0,
                    'countable_late'                 => $EmployeeAttendance ? $EmployeeAttendance->countable_late : 0,
                    'was_absent'                     => $TotlaDayOfMonth - $total_present,
                    'total_late_status'              => $total_late_status,
                    'TotalAcademicDay'               => $TotalAcademicDay  ?? 0,
                    'total_leave'                    => $total_leave  ?? 0,
                    'total_unpaid_leave'             => $total_unpaid_leave  ?? 0,
                    'total_half_day_leave'             => $total_half_day_leave  ?? 0,
                    'total_without_unpaid_leave'     => $total_without_unpaid_leave  ?? 0,
                ];
            });


            if (count($employees) < 1) {
                throw new Exception('No Data Found!');
            }

            return response()->json(
                [
                    'employees'       => $employees,
                    'TotlaDayOfMonth' => $TotlaDayOfMonth,
                ],
                200
            );
        } catch (Exception $exception) {

            return JsonResponse::create(['message' => $exception->getMessage()], 410);
        }
    }
    public function getAllEmployeeReportByDatePayroll(Request $request)
    {
        try {

            ini_set('memory_limit', '-1');

            $startDate  = Carbon::parse($request->startDate)->format('Y-m-d');
            $endDate    =   Carbon::parse($request->EndDate)->format('Y-m-d');

            $search_txt = $request->search_txt;
            $user_type  = $request->types ?? 'employee';
            $status     = $request->status;
            $is_teacher = $request->is_teacher ?? [0, 1];
            $txt        = '%' . $search_txt . '%';

            $employees = $this->employee
                ->with([
                    'payrolls' => function ($query) use ($startDate, $endDate) {
                        $query->select(
                            'user_id',
                            'base_salary',
                            'total_present',
                            'total_absent',
                            'total_late',
                            'late_status_count',
                            'absent_count',
                            'total_bonus',
                            'total_allowance',
                            'total_deduction',
                            'countable_absent',
                            'unpaid_leave_count',
                            'total_unpaid_leave',
                            'countable_unpaid_leave',
                            'countable_late',
                            'year',
                            'month',
                            'pay_date',
                            'created_at',
                        )
                            ->whereBetween('pay_date', [$startDate, $endDate]);
                    }
                ])
                ->select('id', 'username', 'user_type')
                ->where('user_type', $user_type)
                ->when($status == 1, function ($q) use ($status) {
                    return $q->whereHas('activeHistory');
                })
                ->whereHas('empHistories', function ($q1) use ($is_teacher) {
                    return $q1->whereIn('is_teacher', $is_teacher);
                })
                ->when($status < 2, function ($q) use ($status) {
                    return $q->where('status', $status);
                })
                ->when($status == 2, function ($q) {
                    return $q->whereNotNull('deleted_at');
                })
                ->when($search_txt, function ($q) use ($txt) {
                    return $q->where(function ($query) use ($txt) {
                        $query->where('username', 'LIKE', $txt)
                            ->orWhere('email', 'LIKE', $txt)
                            ->orWhere('contact_number', 'LIKE', $txt)
                            ->orWhereTranslationLike('first_name', $txt)
                            ->orWhereTranslationLike('last_name', $txt);
                    });
                })
                ->whereHas('payrolls', function ($query) use ($startDate, $endDate) {
                    $query->whereBetween('pay_date', [$startDate, $endDate]);
                })
                ->orderBy('employee_serial', 'ASC')
                ->get();


            if (count($employees) < 1) {
                throw new Exception('No Data Found!');
            }

            $employees->transform(function ($employe) {
                $payroll = $employe->payrolls->first();


                // Check if $payroll is set before accessing its properties
                if ($payroll) {
                    $deductions = $this->calculateDeductions($payroll);
                    $SalaryofOneDay = $deductions['SalaryofOneDay'];
                    $total_late_deduction = $deductions['total_late_deduction'];
                    $total_absent_deduction = $deductions['total_absent_deduction'];
                    $total_unpaid_leave_deduction = $deductions['total_unpaid_leave_deduction'];
                } else {
                    // If $payroll is not set, set $SalaryofOneDay to 0
                    $SalaryofOneDay = 0;
                    $total_late_deduction = 0;
                    $total_absent_deduction = 0;
                    $total_unpaid_leave_deduction = 0;
                }

                return [
                    'id'                         =>  $employe->id,
                    'full_name'                  =>  $employe->full_name,
                    'username'                   =>  $employe->username,
                    'base_salary'                =>  $payroll ? $payroll->base_salary : 0,
                    'bonus'                      =>  $payroll ? $payroll->total_bonus : 0,
                    'total_present'              =>  $payroll ? $payroll->total_present : 0,
                    'total_absent'               =>  $payroll ? $payroll->total_absent : 0,
                    'allowances'                 =>  $payroll ? $payroll->total_allowance : 0,
                    'late_status'                =>  $payroll ? $payroll->late_status_count : 0,
                    'absent_status'              =>  $payroll ? $payroll->absent_count : 0,



                    'totalCountableLateness'     =>  $result ?? 0,
                    'absent_deductions'          =>  round($total_absent_deduction),
                    'late_deductions'            =>  round($total_late_deduction),
                    'unpaid_leave_deduction'     =>  round($total_unpaid_leave_deduction),
                    'deductions'                 =>  $payroll ? $payroll->total_deduction : 0,
                    'gross_pay'                  =>  $payroll ? $payroll->gross_pay : 0,
                    'net_pay'                    =>  $payroll ? round($payroll->net_pay - ($total_late_deduction + $total_absent_deduction + $total_unpaid_leave_deduction)) : 0,
                    'pay_date'                   =>  $payroll ? Carbon::parse($payroll->created_at)->format('d M Y') : 'null',
                ];
            });


            $totalBase_salary =  number_format($employees->sum('base_salary'));
            $totalBonus       =  number_format($employees->sum('bonus'));
            $total_present    =  number_format($employees->sum('total_present'));
            $total_absent     =  $employees->sum('total_absent');
            $totalAllowances  =  number_format($employees->sum('allowances'));
            $totalGross_pay   =  number_format($employees->sum('gross_pay'));
            $totalDeductions  =  number_format($employees->sum('deductions'));
            $totalNetPay      =  number_format($employees->sum('net_pay'));

            $site_id = Session::get('SITE_ID');
            $site_info =  SiteInfo::select('id', 'site_logo', 'site_phone')->findOrFail($site_id);



            if ($request->pdf_report_download) {
                $pdf = PDF::loadView(
                    'default.admin.site.employee.payroll.site-employees-payroll-date-wise-pdf',
                    compact(
                        'employees',
                        'totalBase_salary',
                        'totalBonus',
                        'total_present',
                        'total_absent',
                        'totalAllowances',
                        'totalGross_pay',
                        'totalDeductions',
                        'totalNetPay',
                        'startDate',
                        'endDate',
                        'site_info',
                    )
                )
                    ->setPaper('a4', 'landscape')
                    ->setWarnings(false);


                $path = storage_path() . '/pdf';
                $time = \Auth::user()->id . '-' . Carbon::now()->unix();
                $pdf->save($path . '/' . $time . '-' . $startDate . '-' . $endDate . '.pdf', 'UTF-8');
                $name = $time . '-' . $startDate . '-' . $endDate . '.pdf';
                $route_name = route('open-pdf-file-storage', [$name]);
            } else {
                $route_name = null;
            }


            return response()->json([
                'employees'        => $employees,
                'totalBase_salary' => $totalBase_salary,
                'totalBonus'       => $totalBonus,
                'total_present'    => $total_present,
                'total_absent'     => $total_absent,
                'totalAllowances'  => $totalAllowances,
                'totalGross_pay'   => $totalGross_pay,
                'totalDeductions'  => $totalDeductions,
                'totalNetPay'      => $totalNetPay,
                'link'             => $route_name,
            ], 200);
        } catch (Exception $exception) {
            return JsonResponse::create(['message' => $exception->getMessage()], 410);
        }
    }
    public function getAllEmployeeReportByMonthVoucherPayroll(Request $request)
    {
        try {

            ini_set('memory_limit', '-1');
            $search_txt       = $request->search_txt;
            $user_type        = $request->types ?? 'employee';
            $status           = $request->status;
            $is_teacher       = $request->is_teacher ?? [0, 1];
            $txt              = '%' . $search_txt . '%';
            $year             = $request->selectedYear;
            $month            = $request->selectedMonth;
            $firstDayOfMonth  = mktime(0, 0, 0, $month, 1, $year);

            $TotlaAcademicDay = date('t', $firstDayOfMonth);

            $targetMonth = Carbon::create($year, $month, 1)->format('Y-m');

            $employees = $this->getEmployeesWithPayrollAndAttendance($targetMonth, $year, $month, $user_type, $status, $is_teacher, $search_txt);

            if (count($employees) < 1) {
                throw new Exception('No Data Found!');
            }

            $employees->transform(function ($employe) {
                return $this->transformEmployeeData($employe);
            });

            $totalBase_salary =  number_format($employees->sum('base_salary'));
            $totalBonus       =  number_format($employees->sum('bonus'));
            $total_present    =  number_format($employees->sum('total_present'));
            $total_absent     =  $employees->sum('total_absent');
            $totalAllowances  =  number_format($employees->sum('allowances'));
            $totalGross_pay   =  number_format($employees->sum('gross_pay'));
            $totalDeductions  =  number_format($employees->sum('deductions'));
            $totalNetPay      =  number_format($employees->sum('net_pay'));



            $site_id = Session::get('SITE_ID');
            $site_info =  SiteInfo::select('id', 'site_logo', 'site_phone')->findOrFail($site_id);

            $pdf = PDF::loadView(
                'default.admin.site.employee..payroll.site-all-employees-payroll-voucher-pdf',
                compact(
                    'employees',
                    'site_info',
                    'year',
                    'month',
                )
            )
                ->setPaper('a4', 'protrit')
                ->setWarnings(false);


            $path = storage_path() . '/pdf';
            $time = \Auth::user()->id . '-' . Carbon::now()->unix();
            $name = $time . 'single' . '-' . $month . '-' . $year . '.pdf';
            $pdf->save($path . '/' . $name, 'UTF-8');


            return response()->json(['employees' => $employees, 'link' => route('open-pdf-file-storage', [$name])], 200);
        } catch (Exception $exception) {
            return JsonResponse::create(['message' => $exception->getMessage()], 410);
        }
    }

    private function getEmployeesWithPayrollAndAttendance($targetMonth, $year, $month, $user_type, $status, $is_teacher, $search_txt = null)
    {
        $txt = '%' . $search_txt . '%';

        return $this->employee
            ->with([
                'payrolls' => function ($query) use ($targetMonth) {
                    $query->select(
                        'user_id',
                        'base_salary',
                        'total_present',
                        'total_absent',
                        'total_late',
                        'late_status_count',
                        'absent_count',
                        'total_bonus',
                        'total_allowance',
                        'total_deduction',
                        'countable_absent',
                        'unpaid_leave_count',
                        'total_unpaid_leave',
                        'total_half_day_leave',
                        'half_day_leave_count',
                        'countable_half_day_leave',
                        'countable_unpaid_leave',
                        'countable_late',
                        'year',
                        'month',
                        'pay_date',
                        'created_at',
                    )
                        ->where(DB::raw('DATE_FORMAT(pay_date, "%Y-%m")'), $targetMonth)
                        ->groupBy('user_id');
                },
                'EmployeeAttendance' => function ($query) use ($year, $month) {
                    $query->select(
                        'user_id',
                        'countable_absent',
                        'countable_late'
                    )
                        ->where('year', $year)
                        ->where('month', $month);
                },
                'Allowances' => function ($query) use ($targetMonth) {
                    $query->select(
                        'user_id',
                        'allowance_amount',
                        'allowance_type',
                        'allowance_date',
                    )
                        ->where(DB::raw('DATE_FORMAT(allowance_date, "%Y-%m")'), $targetMonth);
                },
                'Bonus' => function ($query) use ($targetMonth) {
                    $query->select(
                        'user_id',
                        'bonus_amount',
                        'bonus_type',
                        'bonus_date',
                    )->where(DB::raw('DATE_FORMAT(bonus_date, "%Y-%m")'), $targetMonth);
                },
                'Deductions' => function ($query) use ($targetMonth) {
                    $query->select(
                        'user_id',
                        'deduction_amount',
                        'deduction_type',
                        'deduction_date',
                    )->where(DB::raw('DATE_FORMAT(deduction_date, "%Y-%m")'), $targetMonth);
                }

            ])

            ->select('id', 'username', 'user_type')
            ->where('user_type', $user_type)
            ->when($status == 1, function ($q) {
                return $q->whereHas('activeHistory');
            })
            ->whereHas('empHistories', function ($q1) use ($is_teacher) {
                return $q1->whereIn('is_teacher', $is_teacher);
            })
            ->when($status < 2, function ($q) use ($status) {
                return $q->where('status', $status);
            })
            ->when($status == 2, function ($q) {
                return $q->whereNotNull('deleted_at');
            })
            ->when($search_txt, function ($q) use ($txt) {
                return $q->where(function ($query) use ($txt) {
                    $query->where('username', 'LIKE', $txt)
                        ->orWhere('email', 'LIKE', $txt)
                        ->orWhere('contact_number', 'LIKE', $txt)
                        ->orWhereTranslationLike('first_name', $txt)
                        ->orWhereTranslationLike('last_name', $txt);
                });
            })
            ->whereHas('payrolls', function ($query) use ($targetMonth) {
                $query->where(DB::raw('DATE_FORMAT(pay_date, "%Y-%m")'), $targetMonth);
            })
            ->orderBy('employee_serial', 'ASC')
            ->get();
    }

    public function getAllEmployeeReportByMonthPayroll(Request $request)
    {
        try {

            ini_set('memory_limit', '-1');
            $search_txt       = $request->search_txt;
            $user_type        = $request->types ?? 'employee';
            $status           = $request->status;
            $is_teacher       = $request->is_teacher ?? [0, 1];
            $txt              = '%' . $search_txt . '%';
            $year             = $request->selectedYear;
            $month            = $request->selectedMonth;
            $firstDayOfMonth  = mktime(0, 0, 0, $month, 1, $year);

            $TotlaAcademicDay = date('t', $firstDayOfMonth);

            $targetMonth = Carbon::create($year, $month, 1)->format('Y-m');

            $employees = $this->getEmployeesWithPayrollAndAttendance($targetMonth, $year, $month, $user_type, $status, $is_teacher, $search_txt);

            if (count($employees) < 1) {
                throw new Exception('No Data Found!');
            }

            $employees->transform(function ($employe) {
                return $this->transformEmployeeData($employe);
            });


            $totalBase_salary =  number_format($employees->sum('base_salary'));
            $totalBonus       =  number_format($employees->sum('bonus'));
            $total_present    =  number_format($employees->sum('total_present'));
            $total_absent     =  $employees->sum('total_absent');
            $totalAllowances  =  number_format($employees->sum('allowances'));
            $totalGross_pay   =  number_format($employees->sum('gross_pay'));
            $totalDeductions  =  number_format($employees->sum('deductions'));
            $totalNetPay      =  number_format($employees->sum('net_pay'));


            $site_id = Session::get('SITE_ID');
            $site_info =  SiteInfo::select('id', 'site_logo', 'site_phone')->findOrFail($site_id);

            if ($request->pdf_report_download) {
                $pdf = PDF::loadView(
                    'default.admin.site.employee.payroll.site-employees-payroll-pdf',
                    compact(
                        'employees',
                        'TotlaAcademicDay',
                        'totalBase_salary',
                        'totalBonus',
                        'total_present',
                        'total_absent',
                        'totalAllowances',
                        'totalGross_pay',
                        'totalDeductions',
                        'totalNetPay',
                        'month',
                        'year',
                        'site_info',
                    )
                )
                    ->setPaper('a4', 'landscape')
                    ->setWarnings(false);


                $path = storage_path() . '/pdf';
                $time = \Auth::user()->id . '-' . Carbon::now()->unix();
                $pdf->save($path . '/' . $time . '-' . $month . '-' . $year . '.pdf', 'UTF-8');
                $name = $time . '-' . $month . '-' . $year . '.pdf';
                $route_name = route('open-pdf-file-storage', [$name]);
            } else {
                $route_name = null;
            }


            return [
                'employees'        => $employees,
                'TotlaAcademicDay' => $TotlaAcademicDay,
                'totalBase_salary' => $totalBase_salary,
                'totalBonus'       => $totalBonus,
                'total_present'    => $total_present,
                'total_absent'     => $total_absent,
                'totalAllowances'  => $totalAllowances,
                'totalGross_pay'   => $totalGross_pay,
                'totalDeductions'  => $totalDeductions,
                'totalNetPay'      => $totalNetPay,
                'link'             => $route_name,
            ];
        } catch (Exception $exception) {
            return JsonResponse::create(['message' => $exception->getMessage()], 410);
        }
    }


    private function transformEmployeeData($employe)
    {
        $payroll = $employe->payrolls->first();
        $EmployeeAttendance = $employe->EmployeeAttendance->first() ?? '';

        if ($payroll) {
            $deductions = $this->calculateDeductions($payroll);

            $SalaryofOneDay                 = $deductions['SalaryofOneDay'];
            $total_late_deduction           = $deductions['total_late_deduction'];
            $total_absent_deduction         = $deductions['total_absent_deduction'];
            $total_unpaid_leave_deduction   = $deductions['total_unpaid_leave_deduction'];
            $total_half_day_leave_deduction = $deductions['total_half_day_leave_deduction'];
        } else {
            $SalaryofOneDay                 = 0;
            $total_late_deduction           = 0;
            $total_absent_deduction         = 0;
            $total_unpaid_leave_deduction   = 0;
            $total_half_day_leave_deduction = 0;
        }

        return [
            'id'                       => $employe->id,
            'full_name'                => $employe->full_name,
            'username'                 => $employe->username,

            'Deductions'               => $employe->Deductions,
            'Bonus'                    => $employe->Bonus,
            'Allowances'               => $employe->Allowances,

            'base_salary'              => $payroll ? $payroll->base_salary : 0,
            'bonus'                    => $payroll ? $payroll->total_bonus : 0,
            'total_present'            => $payroll ? $payroll->total_present : 0,
            'total_absent'             => $payroll ? $payroll->total_absent : 0,
            'allowances'               => $payroll ? $payroll->total_allowance : 0,
            'late_status'              => $payroll ? $payroll->late_status_count : 0,
            'absent_status'            => $payroll ? $payroll->absent_count : 0,
            'totalCountableLateness'   => $result ?? 0,
            'absent_deductions'        => round($total_absent_deduction),
            'late_deductions'          => round($total_late_deduction),
            'unpaid_leave_deduction'   => round($total_unpaid_leave_deduction),
            'half_day_leave_deduction' => round($total_half_day_leave_deduction),
            'deductions'               => $payroll ? $payroll->total_deduction : 0,
            'gross_pay'                => $payroll ? $payroll->gross_pay : 0,
            'net_pay'                  => $payroll ? round($payroll->net_pay - ($total_late_deduction + $total_absent_deduction + $total_unpaid_leave_deduction + $total_half_day_leave_deduction)) : 0,
            'pay_date'                 => $payroll ? Carbon::parse($payroll->created_at)->format('d M Y') : 'null',
        ];
    }


    public function getAllEmployeeYearMonthPayroll(Request $request)
    {

        try {


            $search_txt = $request->search_txt;
            $user_type  = $request->types ?? 'employee';
            $status     = $request->status;
            $is_teacher = $request->is_teacher ?? [0, 1];




            $txt             = '%' . $search_txt . '%';
            $year            = $request->selectedYear;
            $month           = $request->selectedMonth;
            $firstDayOfMonth = mktime(0, 0, 0, $month, 1, $year);
            $TotlaDayOfmonth = date('t', $firstDayOfMonth);



            $targetMonth = Carbon::create($year, $month, 1)->format('Y-m');
            $employees = $this->employee
                ->with(['designation'])
                ->with('BaseSalery')
                ->with([
                    'EmployeeAttendance'  => function ($query) use ($year, $month) {
                        $query->where('year', $year)
                            ->where('month', $month);
                    }
                ])
                ->with([
                    'payrolls'  => function ($query) use ($year, $month) {
                        $query->where('year', $year)
                            ->where('month', $month);
                    }
                ])
                ->with([
                    'Allowances' => function ($query) use ($targetMonth) {
                        $query->select(
                            'user_id',
                            DB::raw('SUM(allowance_amount) as total_allowance_amount')
                        )
                            ->where(DB::raw('DATE_FORMAT(allowance_date, "%Y-%m")'), $targetMonth)
                            ->groupBy('user_id');
                    }
                ])
                ->with([
                    'Bonus' => function ($query) use ($targetMonth) {
                        $query->select(
                            'user_id',
                            DB::raw('SUM(bonus_amount) as total_bonus_amount')
                        )
                            ->where(DB::raw('DATE_FORMAT(bonus_date, "%Y-%m")'), $targetMonth)
                            ->groupBy('user_id');
                    }
                ])
                ->with([
                    'Deductions' => function ($query) use ($targetMonth) {
                        $query->select(
                            'user_id',
                            DB::raw('SUM(deduction_amount) as total_deduction_amount')
                        )
                            ->where(DB::raw('DATE_FORMAT(deduction_date, "%Y-%m")'), $targetMonth)
                            ->groupBy('user_id');
                    }
                ])
                ->select('id', 'username', 'user_type')
                ->where('user_type', $user_type)
                ->when($status == 1, function ($q) use ($status) {
                    return $q->whereHas('activeHistory');
                })
                ->whereHas('empHistories', function ($q1) use ($is_teacher) {
                    return $q1->whereIn('is_teacher', $is_teacher);
                })
                ->when($status < 2, function ($q) use ($status) {
                    return $q->where('status', $status);
                })
                ->when($status == 2, function ($q) {
                    return $q->whereNotNull('deleted_at');
                })
                ->when($search_txt, function ($q) use ($txt) {
                    return $q->where(function ($query) use ($txt) {
                        $query->where('username', 'LIKE', $txt)
                            ->orWhere('email', 'LIKE', $txt)
                            ->orWhere('contact_number', 'LIKE', $txt)
                            ->orWhereTranslationLike('first_name', $txt)
                            ->orWhereTranslationLike('last_name', $txt);
                    });
                })
                ->orderBy('employee_serial', 'ASC')
                ->get();
            $employees->transform(function ($value) use ($TotlaDayOfmonth) {

                $EmployeeAttendance     =  $value->EmployeeAttendance->first();
                $total_allowance_amount = $value->Allowances->first()->total_allowance_amount ?? 0;
                $total_bonus_amount     = $value->Bonus->first()->total_bonus_amount ?? 0;
                $base_sallery           = $value->BaseSalery->base_sallery ?? 0;
                $total_deduction_amount = $value->Deductions->first()->total_deduction_amount ?? 0;

                return [
                    'id'                     => $value->id,
                    'full_name'              => $value->full_name,
                    'username'               => $value->username,
                    'user_type'              => $value->user_type,
                    'designation'            => $value->designation->designation_name ?? '',
                    'baseSalery'             => $base_sallery,
                    'was_present'            => $EmployeeAttendance ? ($EmployeeAttendance->was_present + $EmployeeAttendance->manually_attendance + $EmployeeAttendance->off_day + $EmployeeAttendance->total_leave) : 0,
                    'off_day'                => $EmployeeAttendance ? $EmployeeAttendance->off_day : 0,
                    'academic_day'           => $EmployeeAttendance ? $EmployeeAttendance->academic_day : 0,
                    'late_day'               => $EmployeeAttendance ? $EmployeeAttendance->late_day : 0,
                    'absent'                 => $EmployeeAttendance ? $TotlaDayOfmonth - ($EmployeeAttendance->was_present + $EmployeeAttendance->manually_attendance + $EmployeeAttendance->off_day + $EmployeeAttendance->total_leave) : 0,
                    'late_day'               => $EmployeeAttendance ? $EmployeeAttendance->late_day : 0,
                    'countable_absent'       => $EmployeeAttendance ? $EmployeeAttendance->countable_absent : 0,
                    'countable_late'         => $EmployeeAttendance ? $EmployeeAttendance->countable_late : 0,

                    'countable_unpaid_leave' => $EmployeeAttendance ? $EmployeeAttendance->countable_unpaid_leave : 0,
                    'total_leave'            => $EmployeeAttendance ? $EmployeeAttendance->total_leave : 0,

                    'countable_half_day_leave' => $EmployeeAttendance ? $EmployeeAttendance->countable_half_day_leave : 0,
                    'total_half_day_leave'     => $EmployeeAttendance ? $EmployeeAttendance->total_half_day_leave : 0,

                    'total_allowance_amount' => $total_allowance_amount,
                    'total_bonus_amount'     => $total_bonus_amount,
                    'total_deduction_amount' => $total_deduction_amount,
                    'net_pay'                => ($base_sallery + $total_allowance_amount + $total_bonus_amount) - $total_deduction_amount,
                    'payrolls'               => $value->payrolls ?? '',
                    'late_status_count'      => $value->payrolls->first() ? $value->payrolls->first()->late_status_count : 0,
                    'absent_count'           => $value->payrolls->first() ? $value->payrolls->first()->absent_count : 0,
                    'unpaid_leave_count'     => $value->payrolls->first() ? $value->payrolls->first()->unpaid_leave_count : 0,
                    'half_day_leave_count'    => $value->payrolls->first() ? $value->payrolls->first()->half_day_leave_count : 0,
                ];
            });
            // return $employees;


            if (count($employees) < 1) {
                throw new Exception('No Data Found!');
            }


            return response()->json([
                'employees'        => $employees,
                'TotlaDayOfmonth'  => $TotlaDayOfmonth,
            ], 200);
        } catch (Exception $exception) {

            return JsonResponse::create(['message' => $exception->getMessage()], 410);
        }
    }
    public function getAllEmployeeReportByMonthPayrollForAccounts(Request $request)
    {

        try {


            $year = Carbon::now()->format('Y');
            $month = $request->month['id'];
            $employee_id = $request->employee_id;
            $targetMonth = Carbon::create($year, $month, 1)->format('Y-m');

            $employee = $this->employee
                ->with(['designation'])
                ->with([
                    'payrolls'  => function ($query) use ($targetMonth) {
                        $query->select(
                            'user_id',
                            'base_salary',
                            'total_present',
                            'total_absent',
                            'total_late',
                            'late_status_count',
                            'absent_count',
                            'total_bonus',
                            'total_allowance',
                            'total_deduction',
                            'countable_absent',
                            'unpaid_leave_count',
                            'total_unpaid_leave',
                            'countable_unpaid_leave',
                            'countable_late',
                            'year',
                            'month',
                            'pay_date',
                            'created_at',
                        )
                            ->where(DB::raw('DATE_FORMAT(pay_date, "%Y-%m")'), $targetMonth)
                            ->groupBy('user_id');
                    }
                ])
                ->with([
                    'EmployeeAttendance' => function ($query) use ($year, $month) {
                        $query->select(
                            'user_id',
                            'countable_absent',
                            'countable_late',
                        )
                            ->where('year', $year)
                            ->where('month', $month);
                    }
                ])
                ->with([
                    'Allowances' => function ($query) use ($targetMonth) {
                        $query->select(
                            'user_id',
                            'allowance_amount',
                            'allowance_type',
                            'allowance_date',
                        )
                            ->where(DB::raw('DATE_FORMAT(allowance_date, "%Y-%m")'), $targetMonth);
                    }
                ])
                ->with([
                    'Bonus' => function ($query) use ($targetMonth) {
                        $query->select(
                            'user_id',
                            'bonus_amount',
                            'bonus_type',
                            'bonus_date',
                        )->where(DB::raw('DATE_FORMAT(bonus_date, "%Y-%m")'), $targetMonth);
                    }
                ])
                ->with([
                    'Deductions' => function ($query) use ($targetMonth) {
                        $query->select(
                            'user_id',
                            'deduction_amount',
                            'deduction_type',
                            'deduction_date',
                        )->where(DB::raw('DATE_FORMAT(deduction_date, "%Y-%m")'), $targetMonth);
                    }
                ])
                ->whereHas('payrolls', function ($query) use ($targetMonth) {
                    $query->where(DB::raw('DATE_FORMAT(pay_date, "%Y-%m")'), $targetMonth);
                })
                ->select('id', 'username', 'user_type')
                ->find($employee_id);

            if ($employee) {
                $payroll = $employee->payrolls->first();
                $EmployeeAttendance = $employee->EmployeeAttendance->first();


                // Check if $payroll is set before accessing its properties
                if ($payroll) {
                    $deductions = $this->calculateDeductions($payroll);

                    $SalaryofOneDay               = $deductions['SalaryofOneDay'];
                    $total_late_deduction         = $deductions['total_late_deduction'];
                    $total_absent_deduction       = $deductions['total_absent_deduction'];
                    $total_unpaid_leave_deduction = $deductions['total_unpaid_leave_deduction'];
                } else {
                    // If $payroll is not set, set $SalaryofOneDay to 0
                    $SalaryofOneDay = 0;
                    $total_late_deduction = 0;
                    $total_absent_deduction = 0;
                }



                return $data['total_salary'] = $payroll ? round($payroll->net_pay - ($total_late_deduction + $total_absent_deduction + $total_unpaid_leave_deduction)) : 0;
            }
        } catch (Exception $exception) {

            return JsonResponse::create(['message' => $exception->getMessage()], 410);
        }
    }
    public function SingleEmployeeReportByDatePayroll(Request $request)
    {
        $startDate   = Carbon::parse($request->startDate)->format('Y-m-d');
        $endDate     = Carbon::parse($request->EndDate)->format('Y-m-d');
        $employee_id = $request->employee_id;


        $employee = $this->employee
            ->with(['designation'])
            ->with([
                'payrolls'  => function ($query) use ($startDate, $endDate) {
                    $query->select(
                        'user_id',
                        'base_salary',
                        'total_present',
                        'total_absent',
                        'total_late',
                        'late_status_count',
                        'absent_count',
                        'total_bonus',
                        'total_allowance',
                        'total_deduction',
                        'countable_absent',
                        'unpaid_leave_count',
                        'total_unpaid_leave',
                        'countable_unpaid_leave',
                        'countable_late',
                        'year',
                        'month',
                        'pay_date',
                        'created_at',
                    )->whereBetween('pay_date', [$startDate, $endDate]);
                }
            ])
            ->with([
                'Allowances' => function ($query) use ($startDate, $endDate) {
                    $query->select(
                        'user_id',
                        'allowance_amount',
                        'allowance_type',
                        'allowance_date',
                    )->whereBetween('allowance_date', [$startDate, $endDate]);
                }
            ])
            ->with([
                'Bonus' => function ($query) use ($startDate, $endDate) {
                    $query->select(
                        'user_id',
                        'bonus_amount',
                        'bonus_type',
                        'bonus_date',
                    )->whereBetween('bonus_date', [$startDate, $endDate]);
                }
            ])
            ->with([
                'Deductions' => function ($query) use ($startDate, $endDate) {
                    $query->select(
                        'user_id',
                        'deduction_amount',
                        'deduction_type',
                        'deduction_date',
                    )->whereBetween('deduction_date', [$startDate, $endDate]);
                }
            ])
            ->whereHas('payrolls', function ($query) use ($startDate, $endDate) {
                $query->whereBetween('pay_date', [$startDate, $endDate]);
            })
            ->select('id', 'username', 'user_type')
            ->find($employee_id);

        if ($employee) {
            $payroll = $employee->payrolls->first();
            $EmployeeAttendance = $employee->EmployeeAttendance->first();

            // Check if $payroll is set before accessing its properties
            if ($payroll) {
                $deductions = $this->calculateDeductions($payroll);

                $SalaryofOneDay               = $deductions['SalaryofOneDay'];
                $total_late_deduction         = $deductions['total_late_deduction'];
                $total_absent_deduction       = $deductions['total_absent_deduction'];
                $total_unpaid_leave_deduction = $deductions['total_unpaid_leave_deduction'];
            } else {
                // If $payroll is not set, set $SalaryofOneDay to 0
                $SalaryofOneDay = 0;
                $total_late_deduction         = 0;
                $total_absent_deduction       = 0;
                $total_unpaid_leave_deduction = 0;
            }



            $data = [
                'id'                         =>  $employee->id,
                'full_name'                  =>  $employee->full_name,
                'username'                   =>  $employee->username,
                'base_salary'                =>  $payroll ? $payroll->base_salary : 0,
                'bonus'                      =>  $payroll ? $payroll->total_bonus : 0,
                'total_present'              =>  $payroll ? $payroll->total_present : 0,
                'total_absent'               =>  $payroll ? $payroll->total_absent : 0,
                'allowances'                 =>  $payroll ? $payroll->total_allowance : 0,
                'late_status'                =>  $payroll ? $payroll->late_status_count : 0,
                'absent_status'              =>  $payroll ? $payroll->absent_count : 0,
                'totalCountableLateness'     =>  $result ?? 0,
                'absent_deductions'          =>  round($total_absent_deduction),
                'late_deductions'            =>  round($total_late_deduction),
                'unpaid_leave_deduction'     =>  round($total_unpaid_leave_deduction),
                'deductions'                 =>  $payroll ? $payroll->total_deduction : 0,
                'gross_pay'                  =>  $payroll ? $payroll->gross_pay : 0,
                'net_pay'                    =>  $payroll ? round($payroll->net_pay - ($total_late_deduction + $total_absent_deduction + $total_unpaid_leave_deduction)) : 0,
                'pay_date'                   =>  $payroll ? Carbon::parse($payroll->created_at)->format('d M Y') : 'null',
            ];
        }
        $site_id = Session::get('SITE_ID');
        $site_info =  SiteInfo::select('id', 'site_logo', 'site_phone')->findOrFail($site_id);

        $employee_allowances     = $employee->Allowances;
        $employee_bonus          = $employee->Bonus;
        $employee_deductions     = $employee->Deductions;

        $total_allowance_amount  = $employee->Allowances->sum('allowance_amount');
        $total_bonus_amount      = $employee->Bonus->sum('bonus_amount');
        $total_deductions_amount = $employee->Deductions->sum('deduction_amount');

        $pdf = PDF::loadView(
            'default.admin.site.employee.payroll.site-single-employees-date-wise-payroll-pdf',
            compact(
                'data',
                'employee_allowances',
                'employee_bonus',
                'employee_deductions',
                'total_allowance_amount',
                'total_bonus_amount',
                'total_deductions_amount',
                'startDate',
                'endDate',
                'site_info',
            )
        )
            ->setPaper('a4', 'landscape')
            ->setWarnings(false);


        $path = storage_path() . '/pdf';
        $time = \Auth::user()->id . '-' . Carbon::now()->unix();
        $name = $time . 'single' . '-' . $startDate . '-' . $endDate . '.pdf';
        $pdf->save($path . '/' . $name, 'UTF-8');


        return response()->json(['link' => route('open-pdf-file-storage', [$name])], 200);
    }
    public function SingleEmployeeVoucherByMonthPayroll(Request $request)
    {
        $year = $request->selectedYear;
        $month = $request->selectedMonth;
        $employee_id = $request->employee_id;
        $targetMonth = Carbon::create($year, $month, 1)->format('Y-m');

        $employees_data = $this->getAllEmployeeReportByMonthPayroll($request);
        $employees = $employees_data['employees'];

        $data = $employees->where('id', $employee_id)->first();


        $site_id = Session::get('SITE_ID');
        $site_info =  SiteInfo::select('id', 'site_logo', 'site_phone')->findOrFail($site_id);

        $employee_allowances     = $data['Allowances'];
        $employee_bonus          = $data['Bonus'];
        $employee_deductions     = $data['Deductions'];

        $total_allowance_amount  = $employees_data['totalAllowances'] ?? 0;
        $total_bonus_amount      = $employees_data['totalBonus'] ?? 0;
        $total_deductions_amount = $employees_data['totalDeductions'] ?? 0;



        $pdf = PDF::loadView(
            'default.admin.site.employee..payroll.site-single-employees-payroll-voucher-pdf',
            compact(
                'data',
                'employee_allowances',
                'employee_bonus',
                'employee_deductions',
                'total_allowance_amount',
                'total_bonus_amount',
                'total_deductions_amount',
                'month',
                'year',
                'site_info'
            )
        )
            ->setPaper('a4', 'protrit')
            ->setWarnings(false);


        $path = storage_path() . '/pdf';
        $time = \Auth::user()->id . '-' . Carbon::now()->unix();
        $name = $time . 'single' . '-' . $month . '-' . $year . '.pdf';
        $pdf->save($path . '/' . $name, 'UTF-8');


        return response()->json(['link' => route('open-pdf-file-storage', [$name])], 200);
    }
    public function SingleEmployeeReportByMonthPayroll(Request $request)
    {

        $year = $request->selectedYear;
        $month = $request->selectedMonth;
        $employee_id = $request->employee_id;
        $targetMonth = Carbon::create($year, $month, 1)->format('Y-m');


        $employees_data = $this->getAllEmployeeReportByMonthPayroll($request);
        $employees = $employees_data['employees'];
        $data = $employees->where('id', $employee_id)->first();


        $site_id = Session::get('SITE_ID');
        $site_info =  SiteInfo::select('id', 'site_logo', 'site_phone')->findOrFail($site_id);

        $employee_allowances     = $data['Allowances'];
        $employee_bonus          = $data['Bonus'];
        $employee_deductions     = $data['Deductions'];

        $total_allowance_amount  = $employees_data['totalAllowances'] ?? 0;
        $total_bonus_amount      = $employees_data['totalBonus'] ?? 0;
        $total_deductions_amount = $employees_data['totalDeductions'] ?? 0;

        $pdf = PDF::loadView(
            'default.admin.site.employee.payroll.site-single-employees-payroll-pdf',
            compact(
                'data',
                'employee_allowances',
                'employee_bonus',
                'employee_deductions',

                'total_allowance_amount',
                'total_bonus_amount',
                'total_deductions_amount',

                'month',
                'year',
                'site_info',
            )
        )
            ->setPaper('a4', 'landscape')
            ->setWarnings(false);


        $path = storage_path() . '/pdf';
        $time = \Auth::user()->id . '-' . Carbon::now()->unix();
        $name = $time . 'single' . '-' . $month . '-' . $year . '.pdf';
        $pdf->save($path . '/' . $name, 'UTF-8');


        return response()->json(['link' => route('open-pdf-file-storage', [$name])], 200);
    }
    // Payroll end

    //API--------------------------------------------

    //get site user list and role list
    public function getAllEmployeeBySearch(Request $request)
    {

        try {

            $user_type = $request->types ?? 'employee';
            //$is_teacher_or_staff = $request->is_teacher_or_staff;

            $search_txt = $request->search_txt;
            $paginate = (int)$request->paginate;
            $status = $request->status;
            $finger_taken = $request->finger_taken;
            $attendance_sms = $request->attendance_sms;
            $is_teacher = $request->is_teacher ?? [0, 1]; //

            //   dd($status);

            if (!$user_type) {
                throw new Exception('Check At Least One Type!' . $user_type);
            }

            $txt = '%' . $search_txt . '%';

            $employees = $this->employee
                ->with('siteRoles')
                ->with(['latestPhoto'])
                ->with(['designation'])
                ->select('id', 'username', 'user_type', 'email', 'contact_number', 'status', 'finger_taken', 'attendance_sms', 'designation_id', 'signature', 'institute_head')
                ->where('user_type', $user_type)
                ->when($status == 1, function ($q) use ($status) {
                    return $q->whereHas('activeHistory');
                })
                ->whereHas('empHistories', function ($q1) use ($is_teacher) {
                    return $q1->whereIn('is_teacher', $is_teacher);
                })
                ->when($status < 2, function ($q) use ($status) {
                    return $q->where('status', $status);
                })
                ->when($status == 2, function ($q) {
                    return $q->whereNotNull('deleted_at');
                })
                ->when($finger_taken, function ($q) use ($finger_taken) {
                    $finger_taken = $finger_taken == 1 ? $finger_taken : 0;
                    return $q->where('finger_taken', $finger_taken);
                })->when($attendance_sms, function ($q) use ($attendance_sms) {
                    $attendance_sms = $attendance_sms == 1 ? $attendance_sms : 0;
                    return $q->where('attendance_sms', $attendance_sms);
                })
                ->when($search_txt, function ($q) use ($txt) {
                    return $q->where(function ($query) use ($txt) {
                        $query->where('username', 'LIKE', $txt)
                            ->orWhere('email', 'LIKE', $txt)
                            ->orWhere('contact_number', 'LIKE', $txt)
                            ->orWhereTranslationLike('first_name', $txt)
                            ->orWhereTranslationLike('last_name', $txt);
                    });
                })
                ->orderBy('employee_serial', 'ASC')
                ->paginate($paginate);


            if (count($employees) < 1) {
                throw new Exception('No Data Found!');
            }


            return response()->json($employees, 200, []);
        } catch (Exception $exception) {

            return JsonResponse::create(['message' => $exception->getMessage()], 410);
        }
    }
    public function getAllEmployeeBySearchForPayroll(Request $request)
    {

        try {

            $user_type = $request->types ?? 'employee';
            //$is_teacher_or_staff = $request->is_teacher_or_staff;

            $search_txt = $request->search_txt;
            $paginate = (int)$request->paginate;
            $status = $request->status;
            $finger_taken = $request->finger_taken;
            $attendance_sms = $request->attendance_sms;
            $is_teacher = $request->is_teacher ?? [0, 1]; //

            //   dd($status);

            if (!$user_type) {
                throw new Exception('Check At Least One Type!' . $user_type);
            }

            $txt = '%' . $search_txt . '%';

            $employees = $this->employee
                ->with('siteRoles')
                ->with(['latestPhoto'])
                ->with(['designation'])
                ->select('id', 'username', 'user_type', 'email', 'contact_number', 'status', 'finger_taken', 'attendance_sms', 'designation_id', 'signature', 'institute_head')
                ->where('user_type', $user_type)
                ->when($status == 1, function ($q) use ($status) {
                    return $q->whereHas('activeHistory');
                })
                ->whereHas('empHistories', function ($q1) use ($is_teacher) {
                    return $q1->whereIn('is_teacher', $is_teacher);
                })
                ->when($status < 2, function ($q) use ($status) {
                    return $q->where('status', $status);
                })
                ->when($status == 2, function ($q) {
                    return $q->whereNotNull('deleted_at');
                })
                ->when($finger_taken, function ($q) use ($finger_taken) {
                    $finger_taken = $finger_taken == 1 ? $finger_taken : 0;
                    return $q->where('finger_taken', $finger_taken);
                })->when($attendance_sms, function ($q) use ($attendance_sms) {
                    $attendance_sms = $attendance_sms == 1 ? $attendance_sms : 0;
                    return $q->where('attendance_sms', $attendance_sms);
                })
                ->when($search_txt, function ($q) use ($txt) {
                    return $q->where(function ($query) use ($txt) {
                        $query->where('username', 'LIKE', $txt)
                            ->orWhere('email', 'LIKE', $txt)
                            ->orWhere('contact_number', 'LIKE', $txt)
                            ->orWhereTranslationLike('first_name', $txt)
                            ->orWhereTranslationLike('last_name', $txt);
                    });
                })
                ->orderBy('employee_serial', 'ASC')
                ->paginate($paginate);


            if (count($employees) < 1) {
                throw new Exception('No Data Found!');
            }


            return response()->json($employees, 200, []);
        } catch (Exception $exception) {

            return JsonResponse::create(['message' => $exception->getMessage()], 410);
        }
    }

    public function getEmployeeDataBySearch(Request $request)
    {

        try {

            $user_type = $request->types ?? 'employee';
            //$is_teacher_or_staff = $request->is_teacher_or_staff;

            $search_txt = $request->search_txt;
            $paginate = (int)$request->paginate;
            $status = $request->status;
            $finger_taken = $request->finger_taken;
            $attendance_sms = $request->attendance_sms;
            $is_teacher     = $request->is_teacher ?? [0, 1]; //

            //   dd($status);

            if (!$user_type) {
                throw new Exception('Check At Least One Type!' . $user_type);
            }

            $txt = '%' . $search_txt . '%';

            $employees = $this->employee
                ->select('id', 'username', 'user_type', 'email', 'contact_number', 'status', 'finger_taken', 'attendance_sms', 'designation_id', 'signature', 'institute_head')
                ->where('user_type', $user_type)
                ->when($status == 1, function ($q) use ($status) {
                    return $q->whereHas('activeHistory', function ($q) use ($status) {});
                })
                ->whereHas('empHistories', function ($q1) use ($is_teacher) {
                    return $q1->whereIn('is_teacher', $is_teacher);
                })
                ->when($status < 2, function ($q) use ($status) {
                    return $q->where('status', $status);
                })
                ->when($status == 2, function ($q) {
                    return $q->whereNotNull('deleted_at');
                })
                ->when($finger_taken, function ($q) use ($finger_taken) {
                    $finger_taken = $finger_taken == 1 ? $finger_taken : 0;
                    return $q->where('finger_taken', $finger_taken);
                })->when($attendance_sms, function ($q) use ($attendance_sms) {
                    $attendance_sms = $attendance_sms == 1 ? $attendance_sms : 0;
                    return $q->where('attendance_sms', $attendance_sms);
                })
                ->when($search_txt, function ($q) use ($txt) {
                    return $q->where(function ($query) use ($txt) {
                        $query->where('username', 'LIKE', $txt)
                            ->orWhere('email', 'LIKE', $txt)
                            ->orWhere('contact_number', 'LIKE', $txt)
                            ->orWhereTranslationLike('first_name', $txt)
                            ->orWhereTranslationLike('last_name', $txt);
                    });
                })
                ->orderBy('employee_serial', 'ASC')
                ->paginate($paginate);


            $employees->getCollection()->transform(function ($value) {
                return [
                    'id' => $value->id,
                    'full_name' => $value->full_name,
                    'username' => $value->username,
                    'user_type' => $value->user_type,
                    'email' => $value->email,
                    'contact_number' => $value->contact_number,
                    'status' => $value->status,
                    'finger_taken' => $value->finger_taken,
                    'attendance_sms' => $value->attendance_sms,
                    'designation' => $value->designation->designation_name ?? '',
                    'signature' => $value->signature,
                    'institute_head' => $value->institute_head,
                ];
            });


            if (count($employees) < 1) {
                throw new Exception('No Data Found!');
            }

            return response()->json($employees, 200, []);
        } catch (Exception $exception) {

            return JsonResponse::create(['message' => $exception->getMessage()], 410);
        }
    }


    /**
     * @return array|\Illuminate\Http\JsonResponse
     */
    public function getVersionYearShiftForEmployee()
    {


        $the_site_info_setting = SiteInfoSetting::first();

        if (!$the_site_info_setting) {
            return Response::json(['message' => 'No "SiteInfoSetting" Found'], 422);
        }

        $academicShift = explode(',', $the_site_info_setting->shift_ids ?? null);
        $academicShiftList = AcademicShift::whereIn('id', $academicShift)
            ->get();

        $academicYear = explode(',', $the_site_info_setting->academic_year_ids ?? null);
        $academicYearList = AcademicYear::whereIn('id', $academicYear)->get();

        $academicVersionIDs = explode(',', $the_site_info_setting->academic_version_ids);
        $academicVersionList = AcademicVersion::whereIn('id', $academicVersionIDs)->get();

        return [
            $academicVersionList,
            $academicYearList,
            $academicShiftList
        ];
    }


    public function getYearVersionAndShiftBasedDepartmentClass($version_id, $year_id, $shift_id)
    { //for employee


        if (!$version_id || !$year_id || !$shift_id) {
            return Response::json(['message' => 'Select Version, Year And Shift For Class List'], 411);
        }

        $academic_class_list = [];
        $academic_department_list = [];

        $academicDepartment = $this->SiteShiftClassGroupSectionSetting
            ->where('academic_version_id', $version_id)
            ->where('academic_year_id', $year_id)
            ->where('academic_shift_id', $shift_id)
            ->whereNotNull('academic_department_id')
            ->groupBy('academic_class_id')
            ->pluck('academic_department_id')
            ->toArray();


        //dd($academicDepartment);

        if (count($academicDepartment) > 0) {
            $academic_department_list = AcademicDepartment::whereIn('id', $academicDepartment)
                ->get();
        } else {
            $emp_class_ids = SiteEmployeeDepartmentClassSubject::groupBy('academic_class_id')
                ->pluck('academic_class_id')->toArray();

            $academic_class_list = AcademicClass::whereIn('id', $emp_class_ids)
                ->orderBy('serial_no', 'ASC')
                ->get();

            //     dd($emp_class_ids);

            if (count($academic_class_list) == 0) {
                return Response::json(['message' => 'You Have No Class Assigned For The Year! Go: <a href="' . route('site-shift-class-group-section-setting-opening-page') . '">Click</a>'], 411);
            }
        }


        return [$academic_class_list, $academic_department_list];
    }


    public function getSimpleActiveEmployeeList()
    {
        $employees = Employee::where('user_type', 'employee')
            ->where('status', true)
            // ->listsTranslations("user_id")
            // ->withTranslation()
            ->with('latestPhoto')
            ->select(['id', 'employee_id', 'biometric_id', 'username'])->get();
        return response()->json($employees, 200, []);
    }

    public function getAllMonthList()
    {

        $items = [];
        $startMonth = Carbon::now()->startOfYear()->format('M');
        $endMonth = Carbon::now()->endOfYear()->format('M');
        $monthRange = CarbonPeriod::create($startMonth, '1 month', $endMonth);
        foreach ($monthRange as $key => $month) {
            $items[$key]['id'] = $key + 1;
            $items[$key]['name'] = Carbon::parse($month)->format('M');
        }


        // dd($items);
        // $employees = Employee::where('user_type', 'employee')
        //     ->where('status', true)
        //     // ->listsTranslations("user_id")
        //     // ->withTranslation()
        //     ->with('latestPhoto')
        //     ->select(['id', 'employee_id', 'biometric_id', 'username'])->get();
        return response()->json($items, 200);
    }


    public function getTheSubjectAssignedTeacherList(Request $request)
    {

        try {
            $emp_ids = [];


            $data = [];
            $data['academic_period_type_id'] = $request->academic_period_type_id;
            $data['academic_year_id'] = $request->academic_year_id;
            $data['academic_shift_id'] = $request->academic_shift_id;
            $data['emp_list_type'] = $request->emp_list_type;
            $data['academic_department_id'] = $request->academic_department_id;
            $data['academic_class_id'] = $request->academic_class_id;
            $data['academic_subject_id'] = $request->academic_subject_id;


            if (!$data['academic_subject_id']) {
                throw new Error('No Subject Selected!');
            }

            //dd($data);

            if ($data['emp_list_type'] == 1) { //all

                $emp_ids = $this->getTheSubjectAssignedTeacherIds($data);
            } else { //available

                $emp_ids = $this->getAvailableTeacherIdsBySubject($data);
            }

            $emp_list = Employee::whereIn('id', $emp_ids)
                ->select(['id', 'employee_id', 'status'])
                ->whereHas('activeHistory')
                ->where('status', true)
                ->get();

            // dd($emp_list);


            if (count($emp_list) < 1) {
                throw new Error('No Teacher Found For The Subject');
            }

            return $emp_list;
        } catch (Exception $exception) {
            return Response::json(['message' => $exception->getMessage()], 410);
        }
    }


    //REPORTs ========================
    public function getSingleEmployeeFullDetailsReport(Request $request)
    {
        try {


            $employee = $this->employee->with([
                'latestPhoto',
                'gender',
                'religion',
                'designation',
                'presentCountry',
                'permanentCountry',
                'presentDivision',
                'permanentDivision',
                'presentDistrict',
                'permanentDistrict',
                'presentThana',
                'permanentThana',
                'bloodGroup'
            ])
                ->select('id', 'username', 'user_type', 'email', 'contact_number', 'status', 'finger_taken', 'attendance_sms', 'date_of_birth', 'nationality', 'birth_certificate_no', 'designation_id', 'nid_number', 'dept_join_date', 'gender_id', 'religion_id', 'designation_id')
                ->where('id', $request->user_id)
                ->orderBy('employee_serial', 'ASC')
                ->first();

            if ($employee) {
                $employee->EmployeeEducationBackground = EmployeeEducationBackground::with('examination')->where('user_id', $employee->id)->where('status', 1)->get();
                $employee->empClassSubject = SiteEmployeeDepartmentClassSubject::with(['emClass', 'department', 'emSubject'])->where('user_id', $employee->id)->get();
                $employee->empExperiance = EmploymentHistory::where('user_id', $employee->id)->where('status', 1)->get();
                $employee->gender_name = $employee->gender ? $employee->gender->gender_name : null;
                $employee->religion_name = $employee->religion ? $employee->religion->name : null;
                $employee->designation_name = $employee->designation ? $employee->designation->name : null;
            }
            //    return session()->all();


            $view = View::make('default.site.employee.enrolment.report.single-employee-full-details-report', compact('employee'));


            // $path = storage_path() . '/pdf';
            // $time = \Auth::user()->id . '-' . Carbon::now()->unix();
            // $pdf->save($path . '/' . $time . '-' . $time . '-' . $time . '.pdf', 'UTF-8');
            // $name = $time . '-' . $time . '-' . $time . '.pdf';
            // $route_name = route('open-pdf-file-storage', [$name]);



            $contents = $view->render();

            $path = storage_path() . '/pdf';
            $mpdf = new \Mpdf\Mpdf([

                'margin_top' => 7.65,    // 10px ≈ 2.65mm
                'margin_right' => 7.65,
                'margin_bottom' => 7.65,
                'margin_left' => 7.65,
            ]);
            $mpdf->autoScriptToLang = true;
            $mpdf->autoLangToFont = true;
            $mpdf->allow_charset_conversion = false;
            $mpdf->WriteHTML($contents);
            $time = "single-employee-full-details-report" . '-' . Carbon::now()->unix();
            $mpdf->Output($path . '/' . $time . '.pdf', 'F');
            //   $time = $file_name_combination . '-' . Carbon::now()->unix();
            //$mpdf->Output($time . '.pdf', 'D');

            $name = $time . '.pdf';


            return Response::json(['link' => route('open-pdf-file-storage', [$name]), 'message' => 'Successfully Created Report'], 200);


            // return response()->json([
            //     'link'     => $route_name,
            // ], 200);
        } catch (Exception $exception) {
            return Response::json(['message' => $exception->getMessage()], 410);
        }
    }
    public function getEmployeeFullDetailsReport(Request $request)
    {
        try {
            $user_type      = $request->types ?? 'employee';
            $search_txt     = $request->search_txt;
            $paginate       = (int)$request->paginate;
            $status         = $request->status;
            $is_teacher     = $request->is_teacher ?? [0, 1];
            $txt = '%' . $search_txt . '%';



            $employees = $this->employee->with([
                'latestPhoto',
                'gender',
                'religion',
                'designation',
                'presentCountry',
                'permanentCountry',
                'presentDivision',
                'permanentDivision',
                'presentDistrict',
                'permanentDistrict',
                'presentThana',
                'permanentThana',
                'bloodGroup'
            ])
                ->select('id', 'username', 'user_type', 'email', 'contact_number', 'status', 'finger_taken', 'attendance_sms', 'designation_id', 'dept_join_date', 'gender_id', 'religion_id', 'designation_id')
                ->where('user_type', $user_type)
                ->when($status == 1, function ($q) use ($status) {
                    return $q->whereHas('activeHistory', function ($q) use ($status) {});
                })
                ->whereHas('activeHistory', function ($q1) use ($is_teacher) {
                    return $q1->whereIn('is_teacher', $is_teacher);
                })
                ->when($status < 2, function ($q) use ($status) {
                    return $q->where('status', $status);
                })
                ->when($status == 2, function ($q) {
                    return $q->whereNotNull('deleted_at');
                })
                ->when($search_txt, function ($q) use ($txt) {
                    return $q->where(function ($query) use ($txt) {
                        $query->where('username', 'LIKE', $txt)
                            ->orWhere('email', 'LIKE', $txt)
                            ->orWhere('contact_number', 'LIKE', $txt)
                            ->orWhereTranslationLike('first_name', $txt)
                            ->orWhereTranslationLike('last_name', $txt);
                    });
                })
                ->orderBy('employee_serial', 'ASC')
                ->get();

            $employees->transform(function ($employee) {
                $employee->EmployeeEducationBackground = EmployeeEducationBackground::with('examination')->where('user_id', $employee->id)->where('status', 1)->get();
                $employee->empClassSubject             = SiteEmployeeDepartmentClassSubject::with(['emClass', 'department', 'emSubject'])->where('user_id', $employee->id)->get();
                $employee->empExperiance               = EmploymentHistory::where('user_id', $employee->id)->where('status', 1)->get();
                $employee->gender_name                 = $employee->gender      ? $employee->gender->gender_name : null;
                $employee->religion_name               = $employee->religion    ? $employee->religion->name      : null;
                $employee->designation_name            = $employee->designation ? $employee->designation->name   : null;

                return $employee;
            });
            $pdf = PDF::loadView('default.site.employee.enrolment.report.employee-full-details-report', compact('employees'))->setPaper('a4', 'portrit')
                ->setWarnings(false);


            $path = storage_path() . '/pdf';
            $time = \Auth::user()->id . '-' . Carbon::now()->unix();
            $pdf->save($path . '/' . $time . '-' . $time . '-' . $time . '.pdf', 'UTF-8');
            $name = $time . '-' . $time . '-' . $time . '.pdf';
            $route_name = route('open-pdf-file-storage', [$name]);

            return response()->json([
                'link'     => $route_name,
                // 'employee'     => $employees,
            ], 200);
        } catch (Exception $exception) {
            return Response::json(['message' => $exception->getMessage()], 410);
        }
    }
    public function getEmployeeReport(Request $request)
    {
        try {

            $user_type = $request->types ?? 'employee';

            $search_txt = $request->search_txt;
            $status = $request->status;
            $is_teacher = explode(',', $request->is_teacher); //
            $img_status = $request->img_status;
            $subject_status = $request->subject_status;

            // dd($is_teacher);

            if (!$user_type) {
                throw new Exception('Check At Least One Type!' . $user_type);
            }
            $txt = '%' . $search_txt . '%';

            $emp_list = $this->employee
                ->select('id', 'username', 'user_type', 'email', 'contact_number', 'status', 'finger_taken', 'attendance_sms', 'designation_id', 'dept_join_date')
                ->where('user_type', $user_type)
                ->when($status == 1, function ($q) use ($status) {
                    return $q->whereHas('activeHistory', function ($q) use ($status) { // status is already in model function
                    });
                })
                ->with('EmployeeDepartmentClassSubject.emClass', 'EmployeeDepartmentClassSubject.emSubject')
                ->whereHas('activeHistory', function ($q1) use ($is_teacher) {
                    return $q1->whereIn('is_teacher', $is_teacher);
                })
                ->when($status < 2, function ($q) use ($status) {
                    return $q->where('status', $status);
                })
                ->when($status == 2, function ($q) {
                    return $q->whereNotNull('deleted_at');
                })
                ->when($search_txt, function ($q) use ($txt) {
                    return $q->where(function ($query) use ($txt) {
                        $query->where('username', 'LIKE', $txt)
                            ->orWhere('email', 'LIKE', $txt)
                            ->orWhere('contact_number', 'LIKE', $txt)
                            ->orWhereTranslationLike('first_name', $txt)
                            ->orWhereTranslationLike('last_name', $txt);
                    });
                })
                ->orderBy('employee_serial', 'ASC')
                ->get();

            $emp_list->transform(function ($employee) {
                $employee->empClassSubject = $employee->EmployeeDepartmentClassSubject->map(function ($empClassSubject) {
                    return [
                        'academic_class' => $empClassSubject->emClass->class_name ?? '',
                        'academic_subject' => $empClassSubject->emSubject->subject_name ?? '',
                    ];
                });
                return $employee;
            });



            $type = '';
            if (count($is_teacher) > 1) {
                $type = "Teacher & Staff";
            }
            if ($is_teacher[0] == 1) {
                $type = "Teacher";
            }
            if ($is_teacher[0] == 0) {
                $type = " Staff";
            }


            $report_tittle = $type . " Report";
            $report_tittle2 = "";


            //  view('default.site.employee.enrolment.report.employee-report', compact('report_tittle', 'emp_list'));
            $pdf = PDF::loadView('default.site.employee.enrolment.report.employee-report', compact('report_tittle', 'report_tittle2', 'emp_list', 'img_status','subject_status'));
            return $pdf->stream(time() . '-routine.pdf');
        } catch (Exception $exception) {
            return Response::json(['message' => $exception->getMessage()], 410);
        }
    }
}
