<?php

namespace App\Http\Controllers\Site\Student\Attendance;

use App\Models\Attendance\StudentAttendance;
use App\Models\Site\Academic\SitePeriodTypeDuration;
use App\Models\Site\Student\Attendance\StudentAbsentDetail;
use App\Models\Site\Student\Attendance\StudentLeave;
use App\Models\Site\Student\Attendance\StudentLeaveDetail;
use App\Models\Site\Student\Student;
use App\Models\Student\StudentHistory;
use Auth;
use Carbon\Carbon;
use Error;
use Exception;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\ValidationException;
use Illuminate\Http\JsonResponse;
use Response;


class StudentLeaveController extends Controller
{

    private $studentAttendance;
    private $absentDetail;
    private $periodTypeDuration;
    private $studentHistory;
    private $student;
    private $studentLeave;
    private $studentLeaveDetail;

    public function __construct(StudentAttendance $studentAttendance, SitePeriodTypeDuration $periodTypeDuration, StudentHistory $studentHistory, Student $student,
                                StudentAbsentDetail $absentDetail, StudentLeave $studentLeave, StudentLeaveDetail $studentLeaveDetail
    )
    {

        $this->middleware('auth');
        $this->middleware('sitepagechecker');
        $this->studentAttendance = $studentAttendance;
        $this->absentDetail = $absentDetail;
        $this->studentHistory = $studentHistory;
        $this->student = $student;
        $this->periodTypeDuration = $periodTypeDuration;
        $this->absentDetail = $absentDetail;
        $this->studentLeave = $studentLeave;
        $this->studentLeaveDetail = $studentLeaveDetail;
    }


    public function getLeaveList(Request $request)
    {
        try {

            $this->validate($request, [
               // 'apply_date' => 'required',
            ]);

            $data = [];
            $data['academic_version_id'] = $request->academic_version_id == 'null' ? null : $request->academic_version_id;
            $data['academic_year_id'] = $request->academic_year_id == 'null' ? null : $request->academic_year_id;
            $data['academic_shift_id'] = $request->academic_shift_id == 'null' ? null : $request->academic_shift_id;
            $data['academic_department_id'] = $request->academic_department_id == 'null' ? null : $request->academic_department_id;
            $data['academic_class_id'] = $request->academic_class_id == 'null' ? null : $request->academic_class_id;
            $data['academic_class_group_id'] = $request->academic_class_group_id == 'null' ? null : $request->academic_class_group_id;
            $data['academic_section_id'] = $request->academic_section_id == 'null' ? null : $request->academic_section_id;
            $data['academic_session_id'] = $request->academic_session_id == 'null' ? null : $request->academic_session_id;


            $apply_date = $request->apply_date;
            $search_txt = $request->search_txt;
            $status = $request->status;
            $paginate = $request->paginate ?? 20;
            $txt = '%' . $search_txt . '%';


            $day = date('D', strtotime($apply_date));


            $student_leave = $this->studentLeave
                ->whereHas('studentHistory', function ($q) use ($data, $status) {

                    $q->when($data['academic_version_id'], function ($query) use ($data) {
                        return $query->where('academic_version_id', $data['academic_version_id']);
                    })->when($data['academic_year_id'], function ($query) use ($data) {
                        return $query->where('academic_year_id', $data['academic_year_id']);
                    })->when($data['academic_shift_id'], function ($query) use ($data) {
                        return $query->where('academic_shift_id', $data['academic_shift_id']);
                    })->when($data['academic_class_id'], function ($query) use ($data) {
                        return $query->where('academic_class_id', $data['academic_class_id']);
                    })->when($data['academic_department_id'], function ($query) use ($data) {
                        return $query->where('academic_department_id', $data['academic_department_id']);
                    })->when($data['academic_section_id'], function ($query) use ($data) {
                        return $query->where('academic_section_id', $data['academic_section_id']);
                    })->when($data['academic_class_group_id'], function ($query) use ($data) {
                        return $query->where('academic_class_group_id', $data['academic_class_group_id']);
                    })->when($data['academic_session_id'], function ($query) use ($data) {
                        return $query->where('academic_session_id', $data['academic_session_id']);
                    })->where('status', $status);
                })
                ->when($search_txt, function ($qu) use ($txt) {
                    $qu->whereHas('studentHistory', function ($q) use ($txt) {
                        $q->whereHas('student', function ($q) use ($txt) {
                            return $q->where(function ($query) use ($txt) {
                                $query->where('username', 'LIKE', $txt)
                                    ->orWhere('email', 'LIKE', $txt)
                                    ->orWhere('contact_number', 'LIKE', $txt)
                                    ->orWhereTranslationLike('first_name', $txt)
                                    ->orWhereTranslationLike('last_name', $txt);
                            });
                        });
                    });
                })
                ->when($apply_date, function ($q) use ($apply_date) {
                    $q->whereDate('applied_time', $apply_date);
                })
                ->orderBy('applied_time', 'DESC')
                ->paginate($paginate);

            //  return $student_leave;

            $student_leave->getCollection()->transform(function ($value) {


                return [

                    'id' => $value->id ?? null,
                    //Leave
                    'leave' => $value,
                    'username' => $value->studentHistory->student->username ?? 'n/a',
                    'student_roll_number' => $value->studentHistory->student_full_roll_number ?? 'n/a',
                    'full_name' => ($value->studentHistory->student->full_name ?? 'n/a'),
                    'details' => ($value->studentHistory->department->name ?? '') . '-' . ($value->studentHistory->stClass->class_name ?? '') . '-' . ($value->studentHistory->section->section_name ?? '') . '-' . ($value->studentHistory->classGroup->group_name ?? ''),

                    'from_date' => $value->from_date ?? null,
                    'to_date' => $value->to_date ?? null,
                    'leave_status' => $value->leave_status ?? null,
                    'approved_cancelled' => $value->approved_cancelled ?? 0,
                    'applied_time' => $value->applied_time ?? null,
                    'approved_cancelled_time' => $value->approved_cancelled_time ?? null,
                    'approved_cancelled_by' => $value->approvedBy->full_name ?? null,


                ];
            });

            if (count($student_leave) < 1) {
                throw new Error('No Leave Found For: ' . date('d M Y', strtotime($apply_date)));
            }


            return response()->json($student_leave, 200);

        } catch (ValidationException $exception) {
            return JsonResponse::create(['message' => $exception->getMessage(), 'errors' => $exception->validator->getMessageBag()->toArray()], 422);
        } catch (Exception $exception) {
            throw new Error($exception->getMessage());
        }

    }

    //Leave Crud
    public function create(Request $request)
    {


        try {

            $this->validate($request, [
                'from_date' => 'required',
                'to_date' => 'required',
                'total_days' => 'required',
                'student_history_id' => 'required',
            ]);

            $from_date = $request->from_date;
            $to_date = $request->to_date;


            if (date('Y-m-d', strtotime($request->from_date)) > date('Y-m-d', strtotime($request->to_date))) {
                return response()->json(['message' => 'From Date Must Be Less Than To Date'], 411);
            }


            $existed = $this->studentLeave
                ->where(function ($q) use ($from_date, $to_date) {
                    $q->whereBetween('from_date', [$from_date, $to_date])
                        ->orWhereBetween('to_date', [$from_date, $to_date]);
                })
                ->where('leave_status', '!=', 'cancelled')
                ->where('student_history_id', $request->student_history_id)
                ->first();


            if ($existed) {
                return response()->json(['message' => 'An Application IS Existed For The Date Range!'], 412);

            }


            $this->studentLeave = new StudentLeave();

            foreach ($this->studentLeave->ownFields as $k => $ownField) {
                if ($request->{$ownField}) {
                    $this->studentLeave->{$ownField} = $request->{$ownField};
                }
            }


            $this->studentLeave->student_history_id = $request->student_history_id;
            $this->studentLeave->status = $request->status;
            $this->studentLeave->applied_time = Carbon::now()->toDateTimeString();
            $this->studentLeave->save();
            return response()->json(['data' => $this->studentLeave, 'message' => ' Successfully Applied The Application!', 'mode' => 'Applied'], 200);

        } catch (ValidationException $exception) {
            return JsonResponse::create(['message' => $exception->getMessage(), 'errors' => $exception->validator->getMessageBag()->toArray()], 422);
        } catch (Exception $exception) {
            throw new Error($exception->getMessage());
        }
    }

    public function update(Request $request)
    {

        try {

            $this->validate($request, [
                'from_date' => 'required',
                'to_date' => 'required',
                'total_days' => 'required',
            ]);

            $from_date = $request->from_date;
            $to_date = $request->to_date;

            if (date('Y-m-d', strtotime($from_date)) > date('Y-m-d', strtotime($to_date))) {
                return response()->json(['message' => 'From Date Must Be Less Than To Date'], 411);
            }


            $existed = $this->studentLeave
                ->where(function ($q) use ($from_date, $to_date) {
                    $q->whereBetween('from_date', [$from_date, $to_date])
                        ->orWhereBetween('to_date', [$from_date, $to_date]);
                })
                ->where('leave_status', '!=', 'cancelled')
                ->where('id', '!=', $request->id)
                ->where('student_history_id', $request->student_history_id)
                ->first();

            if ($existed) {
                return response()->json(['message' => 'An Application IS Existed For The Date Range!'], 412);
            }

            $the_leave = $this->studentLeave->findOrFail($request->id);


            if ($the_leave->leave_status == 'canceled' || $the_leave->leave_status == 'approved') {
                return response()->json(['message' => 'Application IS Already Approved or Cancelled!'], 412);
            }

            foreach ($the_leave->updateFields as $k => $updateField) {
                if ($request->{$updateField}) {
                    $the_leave->{$updateField} = $request->{$updateField};
                }
            }

            $the_leave->status = $request->status;
            $the_leave->save();
            return response()->json(['data' => $the_leave, 'message' => ' Successfully Updated!', 'mode' => 'Update'], 200);

        } catch (ValidationException $exception) {
            return JsonResponse::create(['message' => $exception->getMessage(), 'errors' => $exception->validator->getMessageBag()->toArray()], 422);
        } catch (Exception $exception) {
            throw new Error($exception->getMessage());
        }
    }

    public function approveCancel(Request $request)
    {

        try {

            $type = $request->type;


            $existed = $this->studentLeave
                ->where('id', $request->id)
                ->first();

            if ($existed->leave_status == 'canceled' || $existed->leave_status == 'approved') {
                return response()->json(['message' => 'Application IS Already Approved or Cancelled!'], 412);
            }

            $the_leave = $this->studentLeave->findOrFail($request->id);


            if ($type == 'approve') {
                $the_leave->leave_status = 'approved';
                $this->setLeaveDetails($the_leave);

            }

            if ($type == 'cancel') {
                $the_leave->leave_status = 'cancelled';
            }

            $the_leave->approved_cancelled_by = Auth::user()->id;
            $the_leave->approved_cancelled_time = Carbon::now()->toDateTimeString();
            $the_leave->approved_cancelled = true;
            $the_leave->save();
            return response()->json(['data' => $the_leave, 'message' => ' Successfully Updated!', 'mode' => 'Update'], 200);

        } catch (Exception $exception) {
            throw new Error($exception->getMessage());
        }
    }


    private function setLeaveDetails($the_leave)
    {

        $from_date = $the_leave->from_date;
        $to_date = $the_leave->to_date;

        while ($from_date <= $to_date) {
            $detail = new StudentLeaveDetail();
            $detail->site_student_attendance_leave_id = $the_leave->id;
            $detail->student_history_id = $the_leave->student_history_id;
            $detail->leave_date = $from_date;
            $detail->save();
            $from_date = date('Y-m-d', strtotime($from_date . ' +1 day'));
        }

    }

    public function delete(Request $request)
    {

        try {

            $existed = $this->studentLeave
                ->where('id', $request->id)
                ->first();

            if ($existed->leave_status == 'canceled' || $existed->leave_status == 'approved') {
                return response()->json(['message' => 'Application IS Already Approved or Cancelled!'], 412);
            }

            $existed->delete();

            return response()->json(['message' => 'The Leave Application Successfully Deleted!', 'mode' => 'Delete'], 200);

        } catch (Exception $exception) {
            throw new Error($exception->getMessage());
        }
    }


}
