<?php

namespace App\Http\Controllers\Site\StudentPanel\Attendance;

use App\Models\Site\Student\Attendance\StudentLeave;
use App\Models\Site\Student\Attendance\StudentLeaveDetail;
use Carbon\Carbon;
use Error;
use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\ValidationException;

class LeaveController extends Controller
{

    private $studentLeave;
    private $studentLeaveDetail;

    public function __construct(StudentLeave $studentLeave, StudentLeaveDetail $studentLeaveDetail
    )
    {

        $this->middleware('auth');
        $this->middleware('siteStudentchecker');

        $this->studentLeave = $studentLeave;
        $this->studentLeaveDetail = $studentLeaveDetail;
    }


    public function getLeaveList(Request $request)
    {
        try {


            $apply_date = $request->apply_date;
            $status = $request->status;
            $paginate = $request->paginate ?? 20;



            $student_leave = $this->studentLeave
                ->whereHas('studentHistory', function ($q) {
                    $q->whereHas('student', function ($q) {
                        $q->where('id', \Auth::user()->id);
                    });
                })
                ->when($apply_date, function ($q) use ($apply_date) {
                    $q->whereDate('applied_time', $apply_date);
                })
                ->orderBy('applied_time', 'DESC')
                ->where('status', $status)
                ->paginate($paginate);


            $student_leave->getCollection()->transform(function ($value) {

                return [

                    'id' => $value->id ?? null,
                    //Leave
                    'leave' => $value,
                    'username' => $value->studentHistory->student->username ?? 'n/a',
                    'student_roll_number' => $value->studentHistory->student_full_roll_number ?? 'n/a',
                    'full_name' => ($value->studentHistory->student->full_name ?? 'n/a'),
                    'details' => ($value->studentHistory->department->name ?? '') . '-' . ($value->studentHistory->stClass->class_name ?? '') . '-' . ($value->studentHistory->section->section_name ?? '') . '-' . ($value->studentHistory->classGroup->group_name ?? ''),

                    'leave_status' => $value->leave_status ?? null,
                    'approved_cancelled' => $value->approved_cancelled ?? 0,
                    'applied_time' => $value->applied_time ?? null,
                    'approved_cancelled_time' => $value->approved_cancelled_time ?? null,
                    'approved_cancelled_by' => $value->approvedBy->full_name ?? null,


                ];
            });

            if (count($student_leave) < 1) {
                throw new Error('No Leave Found For: ' . date('d M Y', strtotime($apply_date)));
            }


            return response()->json($student_leave, 200);

        } catch (ValidationException $exception) {
            return JsonResponse::create(['message' => $exception->getMessage(), 'errors' => $exception->validator->getMessageBag()->toArray()], 422);
        } catch (Exception $exception) {
            throw new Error($exception->getMessage());
        }

    }

    //Leave Crud
    public function create(Request $request)
    {


        try {

            $this->validate($request, [
                'from_date' => 'required',
                'to_date' => 'required',
                'total_days' => 'required',
                'student_history_id' => 'required',
            ]);

            $from_date = $request->from_date;
            $to_date = $request->to_date;


            if (date('Y-m-d', strtotime($request->from_date)) > date('Y-m-d', strtotime($request->to_date))) {
                return response()->json(['message' => 'From Date Must Be Less Than To Date'], 411);
            }


            $existed = $this->studentLeave
                ->where(function ($q) use ($from_date, $to_date) {
                    $q->whereBetween('from_date', [$from_date, $to_date])
                        ->orWhereBetween('to_date', [$from_date, $to_date]);
                })
                ->where('leave_status', '!=', 'cancelled')
                ->where('student_history_id', $request->student_history_id)
                ->first();


            if ($existed) {
                return response()->json(['message' => 'An Application IS Existed For The Date Range!'], 412);

            }


            $this->studentLeave = new StudentLeave();

            foreach ($this->studentLeave->ownFields as $k => $ownField) {
                if ($request->{$ownField}) {
                    $this->studentLeave->{$ownField} = $request->{$ownField};
                }
            }


            $this->studentLeave->student_history_id = $request->student_history_id;
            $this->studentLeave->status = $request->status;
            $this->studentLeave->applied_time = Carbon::now()->toDateTimeString();
            $this->studentLeave->save();
            return response()->json(['data' => $this->studentLeave, 'message' => ' Successfully Applied The Application!', 'mode' => 'Applied'], 200);

        } catch (ValidationException $exception) {
            return JsonResponse::create(['message' => $exception->getMessage(), 'errors' => $exception->validator->getMessageBag()->toArray()], 422);
        } catch (Exception $exception) {
            throw new Error($exception->getMessage());
        }
    }

    public function update(Request $request)
    {

        try {

            $this->validate($request, [
                'from_date' => 'required',
                'to_date' => 'required',
                'total_days' => 'required',
            ]);

            $from_date = $request->from_date;
            $to_date = $request->to_date;

            if (date('Y-m-d', strtotime($from_date)) > date('Y-m-d', strtotime($to_date))) {
                return response()->json(['message' => 'From Date Must Be Less Than To Date'], 411);
            }


            $existed = $this->studentLeave
                ->where(function ($q) use ($from_date, $to_date) {
                    $q->whereBetween('from_date', [$from_date, $to_date])
                        ->orWhereBetween('to_date', [$from_date, $to_date]);
                })
                ->where('leave_status', '!=', 'cancelled')
                ->where('id', '!=', $request->id)
                ->where('student_history_id', $request->student_history_id)
                ->first();

            if ($existed) {
                return response()->json(['message' => 'An Application IS Existed For The Date Range!'], 412);
            }

            $the_leave = $this->studentLeave->findOrFail($request->id);


            if ($the_leave->leave_status == 'cancelled' || $the_leave->leave_status == 'approved') {
                return response()->json(['message' => 'Application IS Already Approved or Cancelled!'], 412);
            }

            foreach ($the_leave->updateFields as $k => $updateField) {
                if ($request->{$updateField}) {
                    $the_leave->{$updateField} = $request->{$updateField};
                }
            }

            $the_leave->status = $request->status;
            $the_leave->save();
            return response()->json(['data' => $the_leave, 'message' => ' Successfully Updated!', 'mode' => 'Update'], 200);

        } catch (ValidationException $exception) {
            return JsonResponse::create(['message' => $exception->getMessage(), 'errors' => $exception->validator->getMessageBag()->toArray()], 422);
        } catch (Exception $exception) {
            throw new Error($exception->getMessage());
        }
    }


    public function delete(Request $request)
    {

        try {

            $existed = $this->studentLeave
                ->where('id', $request->id)
                ->first();

            if ($existed->leave_status == 'cancelled' || $existed->leave_status == 'approved') {
                return response()->json(['message' => 'Application IS Already Approved or Cancelled!'], 412);
            }

            $existed->delete();

            return response()->json(['message' => 'The Leave Application Successfully Deleted!', 'mode' => 'Delete'], 200);

        } catch (Exception $exception) {
            throw new Error($exception->getMessage());
        }
    }
}
